package teamcity

import (
	"fmt"
	"github.com/bxcodec/faker"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"log"
	"os"
)

const (
	tcUsernameEnvName = "TC_USERNAME" // the environment variable name for the teamcity username
	tcPasswordEnvName = "TC_PASSWORD" // the environment variable name for the teamcity password
)

// Represents a configuration object for TeamCity
type Config struct {
	Username string
	Password string
	Logger   *logrus.Logger `faker:"-"`
}

// Creates a new TeamCity Configuration based on environment variables
// Pass the logger for TeamCity Client to use
func NewConfig(logger *logrus.Logger) (*Config, error) {
	tcUsername := os.Getenv(tcUsernameEnvName)
	tcPassword := os.Getenv(tcPasswordEnvName)
	if tcUsername == "" || tcPassword == "" {
		return nil, fmt.Errorf("must provide ENV Variables: %s, %s", tcUsernameEnvName, tcPasswordEnvName)
	}

	return &Config{
		Username: tcUsername,
		Password: tcPassword,
		Logger:   logger,
	}, nil
}

// Returns a mock version of Config - used only for testing!
func NewConfigMock() *Config {
	config := &Config{}
	err := faker.FakeData(config)
	if err != nil {
		log.Fatalf("problem creating fake data: %v", err)
	}
	config.Logger, _ = test.NewNullLogger()
	return config
}
