package teamcity

import (
	"fmt"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"os"
	"testing"
)

func TestNewConfig(t *testing.T) {
	nullLogger, _ := test.NewNullLogger()
	missingEnvError := fmt.Sprintf("must provide ENV Variables: %s, %s", tcUsernameEnvName, tcPasswordEnvName)

	t.Run("raises an error when missing", func (t *testing.T) {
		t.Run("TC_USERNAME", func (t *testing.T) {
			err := os.Setenv(tcPasswordEnvName, "testPass")
			defer os.Unsetenv(tcPasswordEnvName)
			if err != nil {
				t.Fatalf("encountered err: %v", err)
			}

			tcConfig, err := NewConfig(nullLogger)
			assert.EqualError(t, err, missingEnvError)
			assert.Nil(t, tcConfig)
		})

		t.Run("TC_PASSWORD", func (t *testing.T) {
			err := os.Setenv(tcUsernameEnvName, "testUser")
			defer os.Unsetenv(tcUsernameEnvName)
			if err != nil {
				t.Fatalf("encountered err: %v", err)
			}

			tcConfig, err := NewConfig(nullLogger)
			assert.EqualError(t, err, missingEnvError)
			assert.Nil(t, tcConfig)
		})
	})

	t.Run("returns a config with the proper username and password", func (t *testing.T) {
		testUsername := "testUser"
		testPassword := "testPass"
		err := os.Setenv(tcUsernameEnvName, testUsername)
		defer os.Unsetenv(tcUsernameEnvName)
		if err != nil {
			t.Fatalf("encountered err: %v", err)
		}

		err = os.Setenv(tcPasswordEnvName, testPassword)
		defer os.Unsetenv(tcPasswordEnvName)
		if err != nil {
			t.Fatalf("encountered err: %v", err)
		}

		config, err := NewConfig(nullLogger)
		assert.NoError(t, err)
		assert.NotNil(t, config)
		assert.Equal(t, testUsername, config.Username)
		assert.Equal(t, testPassword, config.Password)
		assert.Equal(t, nullLogger, config.Logger)
	})
}
