# codecov instance
resource "aws_instance" "codecov_prod" {
  ami                  = "${var.ami}"
  instance_type        = "c5.9xlarge"
  iam_instance_profile = "${aws_iam_instance_profile.sandstorm_codecov_instance_profile.name}"

  root_block_device {
    volume_type = "gp2"
    volume_size = "${var.volume_size}"
  }

  # Security/Subnet Groups for internal access only
  vpc_security_group_ids = ["${data.terraform_remote_state.vpc_state.twitch_subnets_sg}"]
  subnet_id              = "${data.terraform_remote_state.vpc_state.private_a}"

  tags {
    Name        = "codecov-prod"
    Environment = "prod"
  }
}

# Codecov load balancer
resource "aws_elb" "codecov_elb" {
  name            = "codecov-prod-elb"
  subnets         = ["${data.terraform_remote_state.vpc_state.private_a}"]
  security_groups = ["${data.terraform_remote_state.vpc_state.twitch_subnets_sg}"]
  internal        = true

  # TODO: resolve the access issue for this
  #access_logs {
  #  bucket        = "codecov-twitch-access-logs"
  #  bucket_prefix = "codecov"
  #  interval      = 60
  #}

  listener {
    instance_port      = 5000
    instance_protocol  = "http"
    lb_port            = 443
    lb_protocol        = "https"
    ssl_certificate_id = "${var.ssl_certificate_id}"
  }

  health_check {
    healthy_threshold   = 2
    unhealthy_threshold = 2
    timeout             = 5
    target              = "HTTP:5000/login"
    interval            = 30
  }

  instances                   = ["${aws_instance.codecov_prod.id}"]
  cross_zone_load_balancing   = true
  idle_timeout                = 400
  connection_draining         = true
  connection_draining_timeout = 400

  tags {
    Name = "codecov-prod-elb"
  }
}

#Private Link
resource "aws_lambda_function" "pl_lambda_function" {
  function_name = "${var.app_name}-${var.environment}-PrivateLinkLambda"
  s3_bucket     = "exampleloadbalancer-us-west-2"
  s3_key        = "blog-posts/static-ip-for-application-load-balancer/populate_NLB_TG_with_ALB.zip"
  description   = "Register Application Load Balancer to Network Load Balancer"
  handler       = "populate_NLB_TG_with_ALB.lambda_handler"
  role          = "${aws_iam_role.pl_lambda_iam_role.arn}"
  runtime       = "python2.7"
  timeout       = "300"

  environment {
    variables = {
      ALB_DNS_NAME                      = "${var.internal_alb_dns_name}"
      ALB_LISTENER                      = "443"
      CW_METRIC_FLAG_IP_COUNT           = "True"
      INVOCATIONS_BEFORE_DEREGISTRATION = "3"
      MAX_LOOKUP_PER_INVOCATION         = "50"
      NLB_TG_ARN                        = "${aws_lb_target_group.pl_target_group_https.arn}"
      S3_BUCKET                         = "${aws_s3_bucket.elb_ips_bucket.id}"
    }
  }
}

resource "aws_iam_role" "pl_lambda_iam_role" {
  name = "${var.app_name}-${var.environment}-PrivateLink"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
}
EOF
}

resource "aws_iam_role_policy" "pl_lambda_iam_role_policy" {
  name = "Lambda-ALBasTarget"
  role = "${aws_iam_role.pl_lambda_iam_role.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [{
      "Sid": "LambdaLogging",
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": [
        "arn:aws:logs:*:*:*"
      ]
    },
    {
      "Sid": "S3",
      "Action": [
        "s3:Get*",
        "s3:PutObject",
        "s3:CreateBucket",
        "s3:ListBucket",
        "s3:ListAllMyBuckets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "ELB",
      "Action": [
        "elasticloadbalancing:Describe*",
        "elasticloadbalancing:RegisterTargets",
        "elasticloadbalancing:DeregisterTargets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "CW",
      "Action": [
        "cloudwatch:putMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_lambda_permission" "pl_lambda_permission" {
  action        = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.pl_lambda_function.function_name}"
  principal     = "events.amazonaws.com"
  source_arn    = "${aws_cloudwatch_event_rule.pl_event_rule.arn}"
}

resource "aws_cloudwatch_event_rule" "pl_event_rule" {
  description         = "ScheduledRule"
  schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "pl_event_target" {
  rule = "${aws_cloudwatch_event_rule.pl_event_rule.name}"
  arn  = "${aws_lambda_function.pl_lambda_function.arn}"
}

resource "aws_s3_bucket" "elb_ips_bucket" {
  bucket = "${var.app_name}.${var.environment}.${var.aws_region}.${var.account_id}.elb-ips"
  acl    = "private"
}

resource "aws_lb" "pl_nlb" {
  name                             = "${var.app_name}-${var.environment}-PLELB"
  internal                         = true
  load_balancer_type               = "network"
  enable_cross_zone_load_balancing = true

  subnets = [
    "${var.subnet_a}",
  ]
}

resource "aws_lb_target_group" "pl_target_group_https" {
  name                 = "${var.app_name}-${var.environment}-PLTG-https"
  port                 = 443
  protocol             = "TCP"
  vpc_id               = "${var.vpc}"
  target_type          = "ip"
  deregistration_delay = 300

  health_check {
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }
}

resource "aws_lb_listener" "pl_listener_https" {
  load_balancer_arn = "${aws_lb.pl_nlb.arn}"
  port              = "443"
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = "${aws_lb_target_group.pl_target_group_https.arn}"
  }
}

resource "aws_vpc_endpoint_service" "vpc_endpoint_service" {
  acceptance_required        = false
  network_load_balancer_arns = ["${aws_lb.pl_nlb.arn}"]

  allowed_principals = [
    "arn:aws:iam::248953866850:root",
    "arn:aws:iam::153570226312:root",
    "arn:aws:iam::047062552417:root",
  ]
}

output "private_ip" {
  value = "${aws_instance.codecov_prod.private_ip}"
}
