package main

import (
	"code.justin.tv/qe/contacts-service/rpc/contacts"
	"context"
	"fmt"
	"net/http"
	"strconv"
	"time"
	"encoding/json"
	"strings"
	"log"
)

// Mock code cov server
type MockCodeCovServer struct {}

// dummy response for useCaseing
func (s *MockCodeCovServer) getCodeCov() [5]map[string]string {
	return [5]map[string]string{
		{"repo": "video/player-core", "owner": "purushen", "coverage": "50"},
		{"repo": "video/player-ui", "owner": "aisaac", "coverage": "60"},
		{"repo": "qe/contacts-service", "owner": "lukemng", "coverage": "70"},
		{"repo": "qe/heimdall", "owner": "dylan", "coverage": "80"},
		{"repo": "chat/tmi", "owner": "lukemng", "coverage": "60"},
	}
}

type ServiceCatalogMessage struct {
	Data struct {
		Services []struct {
			Id     string `json:"id"`
			Name   string `json:"name"`
			TeamId string `json:"team_id"`
			PrimaryOwner struct {
				Cn  string `json:"cn"`
				Uid string `json:"uid"`
			} `json:"primary_owner"`
		} `json:"services"`
	} `json:"data"`
}

// Lightweight service catalog client
type ServiceCatalog struct {}

func (s *ServiceCatalog) getServices() ServiceCatalogMessage {
	payload := strings.NewReader(`{"operationName":"FetchAllServices","variables":{},"query":"query FetchAllServices {\n  services {\n    ...serviceShallowData\n    }\n}\n\nfragment serviceShallowData on Service {\n  id\n  name\n  team_id\n  primary_owner{\n cn\n uid\n}\n \n    }\n"}`)
	serviceCatalogResponse := ServiceCatalogMessage{}
	for i := 0; i < 3; i++ {
		res, err := http.Post("https://status.internal.justin.tv/api/v2/query", "application/x-www-form-urlencoded", payload)
		if err != nil {
			println("could not retrieve list of services from service catalog")
		}
		json.NewDecoder(res.Body).Decode(&serviceCatalogResponse)
		if len(serviceCatalogResponse.Data.Services) > 0 {
			break
		}
		time.Sleep(500 * time.Millisecond)
	}
	return serviceCatalogResponse
}

const contactsServiceUrl = "https://contacts-api.internal.justin.tv"

func useCaseGetStructure() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "lukemng"
	result, err := api.GetStructure(context.Background(), &contacts.UserRequest{UserId: userId})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	log.Printf("%s's org structure: BU=%s, Org=%s, Team=%s", userId, result.Bu.Name, result.Org.Name, result.Team.Name)
}

func useCaseListReports() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "chris"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth:1})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Persons {
		log.Printf("%s - %s's report = %s (location: %s)", e.PreferredName, userId, e.UserId, e.Location.BuildingName)
	}
}

func useCaseGetManager() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "lukemng"
	result, err := api.GetManager(context.Background(), &contacts.UserRequest{UserId: userId})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	log.Printf("%s's manager = %s", userId, result.Person.UserId)
}

func useCaseListBUs() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	result, err := api.ListBUs(context.Background(), &contacts.BUsRequest{})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Bus {
		log.Printf("bu id=%d, name=%s, desc=%s", e.Id, e.Name, e.Desc)
	}
	if len(result.Bus) == 0 {
		log.Fatalf("could not find bus")
	}
}

func useCaseListOrgs() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	result, err := api.ListOrgs(context.Background(), &contacts.OrgsRequest{})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Orgs {
		log.Printf("org id=%d, name=%s, desc=%s", e.Id, e.Name, e.Desc)
	}
	if len(result.Orgs) == 0 {
		log.Fatalf("could not find orgs")
	}
}

func useCaseListOrgsSpecifiedBU() {
	buId := uint32(1)
	log.Printf("--- bu id = %d ---", buId)
	contactsServer := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	result, err := contactsServer.ListOrgs(context.Background(), &contacts.OrgsRequest{BuId: buId})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Orgs {
		log.Printf("org id=%d, name=%s, desc=%s", e.Id, e.Name, e.Desc)
	}
	if len(result.Orgs) == 0 {
		log.Fatalf("could not find orgs in bu %d", buId)
	}
}

func useCaseListTeams() {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	result, err := api.ListTeams(context.Background(), &contacts.TeamsRequest{})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Teams {
		log.Printf("team id=%d, name=%s, desc=%s", e.Id, e.Name, e.Desc)
	}
	if len(result.Teams) == 0 {
		log.Fatalf("could not find teams")
	}
}

func useCaseListTeamsSpecifiedOrg() {
	orgId := uint32(1)
	log.Printf("--- org id = %d ---", orgId)
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	result, err := api.ListTeams(context.Background(), &contacts.TeamsRequest{OrgId: orgId})
	if err != nil {
		log.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Teams {
		log.Printf("team id=%d, name=%s, desc=%s)", e.Id, e.Name, e.Desc)
	}
	if len(result.Teams) == 0 {
		log.Fatalf("could not find teams in org %d", orgId)
	}
}

func average(entries []int) int {
	sum := 0
	for _, val := range entries {
		sum += val
	}
	return sum / len(entries)
}

func useCaseSimulateCodeCovRollUp() {
	// useCase data - imaginary codecov data
	codeCovServer := &MockCodeCovServer{}
	codecovData := codeCovServer.getCodeCov()

	teamSummary := make(map[string][]int) // team name -> code coverage entries
	orgSummary := make(map[string][]int)  // org name -> code coverage entries
	buSummary := make(map[string][]int)   // bu name -> code coverage entries

	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})

	// illustrate grouping code coverage metrics by teams, orgs, BUs by querying contacts service for org structure
	for _, data := range codecovData {
		// ask contacts-service: what is the org structure of the repo owner
		result, err := api.GetStructure(context.Background(), &contacts.UserRequest{UserId: data["owner"]})
		if err != nil {
			log.Fatalf("fail due to %s", err.Error())
		}

		teamKey := fmt.Sprintf("%s | %s | %s", result.Bu.Name, result.Org.Name, result.Team.Name)
		orgKey := fmt.Sprintf("%s | %s", result.Bu.Name, result.Org.Name)
		buKey := result.Bu.Name

		coverage, err := strconv.Atoi(data["coverage"])
		if err != nil {
			log.Printf("Could not convert to int: %s", data["coverage"])
			continue
		}

		teamSummary[teamKey] = append(teamSummary[teamKey], coverage)
		orgSummary[orgKey] = append(orgSummary[orgKey], coverage)
		buSummary[buKey] = append(buSummary[buKey], coverage)
	}

	// print out the rollups (average per group)
	log.Printf("--- team level rollups ---")
	for team, entries := range teamSummary {
		// print out team -> average for the team
		log.Printf("%s (%d repos): %d%%", team, len(entries), average(entries))
	}
	log.Println()

	log.Printf("--- org level rollups ---")
	for org, entries := range orgSummary {
		// print out team -> average for the team
		log.Printf("%s (%d repos): %d%%", org, len(entries), average(entries))
	}
	log.Println()

	log.Printf("--- bu level rollups ---")
	for bu, entries := range buSummary {
		// print out team -> average for the team
		log.Printf("%s (%d repos): %d%%", bu, len(entries), average(entries))
	}
}

func printOwnedServices(ownerName string, serviceCatalogData ServiceCatalogMessage) {
	var services []string
	for _, service := range serviceCatalogData.Data.Services {
		if service.PrimaryOwner.Uid == ownerName {
			services = append(services, service.Name)
		}
	}
	for _, service := range services {
		log.Printf("service = %s (owner: %s)", service, ownerName)
	}
}


func useCaseSimulateFindAllServicesUnderLeader() {
	//userId := "marty"
	//userId := "mweiler"
	userId := "chris"
	log.Printf("--- all services under %s ---", userId)

	contactsService := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})

	// get services from service catalog
	serviceCatalogService := &ServiceCatalog{}
	serviceCatalogResponse := serviceCatalogService.getServices()
	
	anyLevelReports, err := contactsService.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 10, IncludeManager: true})
	if err != nil {
		log.Fatalf("Failed to get reports from contacts service")
	}

	for _, person := range anyLevelReports.Persons {
		printOwnedServices(person.UserId, serviceCatalogResponse)
	}
}

func main() {
	useCaseGetStructure()
	useCaseListReports()
	useCaseGetManager()
	useCaseListBUs()
	useCaseListOrgs()
	useCaseListOrgsSpecifiedBU()
	useCaseListTeams()
	useCaseListTeamsSpecifiedOrg()
	useCaseSimulateCodeCovRollUp()
	useCaseSimulateFindAllServicesUnderLeader()
}
