package main

import (
	d "code.justin.tv/qe/contacts-service/model/db"
	"code.justin.tv/qe/twitchldap"
	"flag"
	"github.com/jinzhu/gorm"
	_ "github.com/jinzhu/gorm/dialects/postgres"
	"log"
	"os"
	"github.com/nimajalali/go-force/force"
	"code.justin.tv/qe/contacts-service/model/idm"
	"strconv"
	"github.com/aws/aws-lambda-go/lambda"
	cs "code.justin.tv/qe/contacts-service/internal/contactsserver"
	"fmt"
	"reflect"
)

const unknown = "N/A"

func safeCreate(tx *gorm.DB, entity interface{}, entityToId map[string]uint) (uint, error) {
	// 1) if ID already known for the entity instance, set the instance's "ID" field
	// 2) otherwise, create a db record for it, and updates entityToId map with its ID
	entityVal := reflect.Indirect(reflect.ValueOf(entity))
	key := fmt.Sprintf("%#v", entityVal)

	log.Printf("safe-creating %s", key)
	if id, ok := entityToId[key]; ok {
		log.Printf("key %s exists in cache. will set entity's to %d", key, id)
		pval := reflect.ValueOf(entity)
		val := pval.Elem()
		vfld := val.FieldByName("ID")
		vfld.SetUint(uint64(id))

		return id, nil
	} else {
		var err error
		dbId := uint(0)

		switch v := entityVal.Interface().(type) {
		case d.Location:
			err = tx.Create(&v).Error
			dbId = v.ID
		case d.Team:
			err = tx.Create(&v).Error
			dbId = v.ID
		case d.Org:
			err = tx.Create(&v).Error
			dbId = v.ID
		case d.BU:
			err = tx.Create(&v).Error
			dbId = v.ID
		case d.Person:
			err = tx.Create(&v).Error
			dbId = v.ID
		default:
			log.Printf("warning - ignoring unsupported type: %s", v)
		}

		log.Printf("\tcreated db record with id=%d", dbId)

		if err != nil {
			tx.Rollback()
			log.Printf("\terror occured: %s", err)
			return 0, err
		}

		entityToId[key] = dbId
		log.Printf("\tadded to cache, key=%s, value=%d", key, dbId)
		return dbId, nil
	}
}

func findLDAPUser(allUsers []*twitchldap.EmployeeInfo, employeeNumber uint32) *twitchldap.EmployeeInfo {
	for _, e := range allUsers {
		if e.EmployeeNumber == employeeNumber {
			return e
		}
	}
	return nil
}

type OrgStruct struct {
	teamName string
	orgName  string
	buName   string
}

func getBuildingMap() map[string]string {
	forceApi, err := getForce()
	if err != nil {
		log.Fatalf("could not instantiate salesforce api due to: %s", err.Error())
	}

	buildingIdToName := make(map[string]string) // building id (salesforce) => name

	bDesc, err := forceApi.DescribeSObject(&idm.Building{})
	if err != nil {
		log.Fatalf(err.Error())
	}
	bs := &idm.BuildingQueryResponse{}
	err = forceApi.QueryAll(force.BuildQuery(bDesc.AllFields, "Building__c", nil), bs)
	if err != nil {
		log.Fatalf(err.Error())
	}
	for _, b := range bs.Records {
		buildingIdToName[b.Id] = b.Name
	}

	return buildingIdToName
}

// getSfDeptIdToOrgCoordinates() maps Salesforce Department ID to Contacts Service domain (BU, Org, Team)
func getSfDeptIdToOrgCoordinates() map[string]*OrgStruct {
	forceApi, err := getForce()
	if err != nil {
		log.Fatalf("could not instantiate salesforce api due to: %s", err.Error())
	}

	sfDeptIdToOrgCoords := make(map[string]*OrgStruct) // dept id (salesforce) => (buName, orgName, teamName)

	buMap := make(map[string]idm.BusinessUnit)
	buDesc, err := forceApi.DescribeSObject(&idm.BusinessUnit{})
	if err != nil {
		log.Fatalf(err.Error())
	}
	bus := &idm.BusinessUnitQueryResponse{}
	err = forceApi.QueryAll(force.BuildQuery(buDesc.AllFields, "BusinessUnit__c", nil), bus)
	if err != nil {
		log.Fatalf(err.Error())
	}
	for _, b := range bus.Records {
		buMap[b.Id] = b
	}
	log.Printf("total bu records=%d", len(buMap))

	orgMap := make(map[string]idm.Organization)
	orgDesc, err := forceApi.DescribeSObject(&idm.Organization{})
	if err != nil {
		log.Fatalf(err.Error())
	}
	orgs := &idm.OrganizationQueryResponse{}
	err = forceApi.QueryAll(force.BuildQuery(orgDesc.AllFields, "Organization__c", nil), orgs)
	if err != nil {
		log.Fatalf(err.Error())
	}
	for _, o := range orgs.Records {
		orgMap[o.Id] = o
	}
	log.Printf("total org records=%d", len(orgs.Records))

	deptDesc, err := forceApi.DescribeSObject(&idm.Department{})
	if err != nil {
		log.Fatalf(err.Error())
	}
	depts := &idm.DepartmentQueryResponse{}
	err = forceApi.QueryAll(force.BuildQuery(deptDesc.AllFields, "Department__c", nil), depts)
	if err != nil {
		log.Fatalf(err.Error())
	}
	log.Printf("total department records=%d", len(depts.Records))

	for _, dept := range depts.Records { // traverse the Department__c records from salesforce
		if _, ok := sfDeptIdToOrgCoords[dept.Id]; !ok { // if map doesn't yet contain the dept Id, then proceed
			if org, ook := buMap[dept.BusinessUnit]; ook { // if org is found, then proceed
				if bu, bok := orgMap[org.OrganizationId]; bok { // if bu is found, then proceed
					orgStruct := &OrgStruct{
						teamName: dept.CuratedName,
						orgName:  org.CuratedName,
						buName:   bu.CuratedName,
					}
					sfDeptIdToOrgCoords[dept.Id] = orgStruct
				} else {
					log.Printf("org.CuratedName %s not found in orgMap", org.CuratedName)
				}
			} else {
				// Salesforce backend sets dept.BusinessUnit to empty for non-Twitch (ie, Amazon) BUs
				// Note: as of 7/2018, the only matching case is "Twitch Prime"
				log.Printf("dept.BusinessUnit %s (name=%s, id=%s) not found in buMap", dept.BusinessUnit,
					dept.CuratedName, dept.Id)
				orgStruct := &OrgStruct{
					teamName: dept.Name,
					orgName:  unknown,
					buName:   unknown,
				}
				sfDeptIdToOrgCoords[dept.Id] = orgStruct
				log.Printf("Added %v with key %s into sfDeptIdToOrgCoords", orgStruct, dept.Id)
			}
		} else {
			log.Printf("dept.Id %s not found in sfDeptIdToOrgCoords", dept.Id)
		}
	}

	return sfDeptIdToOrgCoords
}

func getForce() (*force.ForceApi, error) {
	ss, err := cs.SetupSandstorm()
	if err != nil {
		log.Printf("warning - unable to init sandstorm client. ignore if on local dev env.")
	}
	clientId := cs.GetEnvVarOrSecrets(ss, "sf_client_id")
	clientSecret := cs.GetEnvVarOrSecrets(ss, "sf_client_secret")
	userName := cs.GetEnvVarOrSecrets(ss, "sf_user")
	password := cs.GetEnvVarOrSecrets(ss, "sf_password")
	secToken := cs.GetEnvVarOrSecrets(ss, "sf_sec_token")
	environ := "production" // this value doesn't matter
	return force.Create("v38.0", clientId, clientSecret, userName, password, secToken, environ)
}

func ingest() error {
	forceApi, err := getForce()
	if err != nil {
		log.Printf("could not instantiate salesforce api due to: %s", err.Error())
		return err
	}
	ldapClient, err := twitchldap.NewClient()
	if err != nil {
		log.Printf("Failed to init ldap client due to: %s", err.Error())
		return err
	}
	allLdapUsers, err := ldapClient.GetAllUserInfo()
	if err != nil {
		log.Printf("Failed to get all users from ldap client due to: %s", err.Error())
		return err
	}

	contactDesc, err := forceApi.DescribeSObject(&idm.Contact{})
	if err != nil {
		log.Printf(err.Error())
		return err
	}
	contacts := &idm.ContactQueryResponse{}
	err = forceApi.QueryAll(force.BuildQuery(contactDesc.AllFields, "Contact", nil), contacts)
	if err != nil {
		log.Printf("could not query Contact data from salesforce due to: %s", err.Error())
		return err
	}

	buildingMap := getBuildingMap()
	orgStructMap := getSfDeptIdToOrgCoordinates()
	sfIdToContacts := make(map[string]idm.Contact)        // salesforce ID => contact
	sfIdToPersons := make(map[string]d.Person)            // salesforce ID => person
	deptIdToTeam := make(map[string]d.Team)               // department id => team
	buildingIdToLocation := make(map[string]d.Location)   // building id => location

	log.Printf("total contacts=%d", len(sfIdToPersons))

	ss, err := cs.SetupSandstorm()
	if err != nil {
		log.Printf("warning - unable to init sandstorm client")
	}
	dbUrl := cs.GetEnvVarOrSecrets(ss, "contacts_db_url")
	db, err := gorm.Open("postgres", dbUrl)
	if err != nil {
		log.Fatalf("Could not connect to database: %s", err)
		return err
	}

	// boilerplate protection per http://gorm.io/docs/transactions.html
	tx := db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	if tx.Error != nil {
		return err
	}

	tx.SingularTable(true)
	tx.AutoMigrate(&d.BU{}, &d.Org{}, &d.Team{}, &d.Person{}, &d.Location{})
	tx.Unscoped().Delete(d.Person{})
	tx.Unscoped().Delete(d.Team{})
	tx.Unscoped().Delete(d.Org{})
	tx.Unscoped().Delete(d.BU{})
	tx.Unscoped().Delete(d.Location{})

	entityToId := make(map[string]uint) // keep track of IDs to prevent recreating multiple records in the same transaction

	// first-pass: persist self
	for _, contact := range contacts.Records {
		if len(contact.Class) != 1 { // valid values: F, T, W, I, B. invalid values: "", "_service*"
			log.Printf("Skipping non-human account type %s for %s", contact.Class, contact.Name)
			continue
		}
		sfIdToContacts[contact.SalesforceId] = contact

		o := orgStructMap[contact.DepartmentId]
		if o == nil { // ignore accounts (service accounts, yet-to-be-pruned test accounts, etc) not part of an org
			log.Printf("Could not find org associated with department id %s for user %s", contact.DepartmentId, contact.PreferredName)
			continue
		}

		employeeNum, err := strconv.Atoi(contact.EmployeeId)
		if err != nil { // this should not happen
			log.Printf("failed to convert employee id to int: %s", contact.EmployeeId)
			continue
		}

		ldapUser := findLDAPUser(allLdapUsers, uint32(employeeNum))
		if ldapUser == nil { // this should not happen
			log.Printf("Could not find LDAP user id for user %s", contact.PreferredName)
			continue //TODO: review this scenario with salesforce team
		}

		bu := d.BU{ID: 0, Name: o.buName, Desc: o.buName}
		_, err = safeCreate(tx, &bu, entityToId)
		if err != nil {
			return err
		}

		org := d.Org{Name: o.orgName, Desc: o.orgName, BUId: bu.ID}
		_, err = safeCreate(tx, &org, entityToId)
		if err != nil {
			return err
		}

		team := d.Team{Name: o.teamName, Desc: o.teamName, OrgId: org.ID}
		teamId, err := safeCreate(tx, &team, entityToId)
		if err != nil {
			return err
		}
		team.ID = teamId
		deptIdToTeam[contact.DepartmentId] = team

		buildingName := buildingMap[contact.BuildingId]
		location := d.Location{BuildingName: buildingName}
		locId, err := safeCreate(tx, &location, entityToId)
		if err != nil {
			return err
		}
		location.ID = locId
		buildingIdToLocation[contact.BuildingId] = location

		employeeNumInt, err := strconv.Atoi(contact.EmployeeId)
		if err != nil { // should not happen
			log.Printf("Failed to convert string employeeNum to int")
			continue
		}
		person := d.Person{
			EmployeeNumber: uint32(employeeNumInt),
			UserId:         ldapUser.UID,
			FirstName:      contact.Firstname,
			LastName:       contact.Lastname,
			PreferredName:  contact.PreferredName,
			TeamId:         deptIdToTeam[contact.DepartmentId].ID,
			Title:          contact.Title,
			TwitchAmznUid:  contact.AmazonUsername,
			Email:          contact.Email,
			EmployeeType:   contact.Class,
			LocationId:     buildingIdToLocation[contact.BuildingId].ID,
			IsActive:       !contact.IsDeleted,
			IsManager:      false,
		}
		personId, err := safeCreate(tx, &person, entityToId)
		if err != nil {
			return err
		}
		person.ID = personId
		log.Printf("created person: (id=%d, name=%s title=%s team=%s)", person.ID, person.PreferredName, person.Title, deptIdToTeam[contact.DepartmentId].Name)
		sfIdToPersons[contact.SalesforceId] = person
	}

	// second pass: set manager ids
	for sfId, person := range sfIdToPersons {
		manager := sfIdToPersons[sfIdToContacts[sfId].ReportsToId]
		log.Printf("Updating %s's manager id to %s (dbid=%d)", person.UserId, manager.UserId, manager.ID)
		tx.Model(&person).Updates(d.Person{
			ManagerId: manager.ID,
		})
		tx.Model(&manager).UpdateColumn(d.Person{IsManager: true})
	}

	tx.Commit()

	return nil
}

func dump() error {
	db, err := gorm.Open("postgres", os.Getenv("CONTACTS_DB_URL"))
	if err != nil {
		log.Printf("Could not connect to database due to " + err.Error())
		return err
	}
	defer db.Close()

	db.SingularTable(true)
	//db.LogMode(true)

	// print db data
	var bus []d.BU
	db.Find(&bus)
	for _, bu := range bus {
		log.Printf("BU=%s", bu.Name)
		var orgs []d.Org
		db.Find(&orgs, d.Org{BUId: bu.ID})
		for _, org := range orgs {
			log.Printf("\tOrg=%s", org.Name)
			var teams []d.Team
			db.Find(&teams, d.Team{OrgId: org.ID})
			for _, team := range teams {
				log.Printf("\t\tTeam=%s", team.Name)
				var persons []d.Person
				db.Find(&persons, d.Person{TeamId: team.ID})
				for _, person := range persons {
					manager := d.Person{}
					db.First(&manager, person.ManagerId)
					log.Printf("\t\t\tPerson=%s %s (%s) Manager=%s %s", person.FirstName, person.LastName, person.Title, manager.FirstName, manager.LastName)
				}
			}
		}
	}
	return nil
}

func main() {
	if len(os.Args) == 1 {
		log.Printf("No command-line arguments passed. Running in AWS lambda mode... " +
			"Pass either -import or -dump if you want to run locally")
		lambda.Start(LambdaHandler) // register LambdaHandler() as a lambda function that AWS can invoke via some trigger
		return
	}

	importFlag := flag.Bool("import", true, "import data")
	dumpFlag := flag.Bool("dump", true, "dump data")
	flag.Parse()

	if *importFlag {
		log.Printf("Starting ingestion...")
		err := ingest()
		if err != nil {
			log.Printf("Data ingestion error: %s", err.Error())
		}
	}
	if *dumpFlag {
		log.Printf("Starting data dump...")
		err := dump()
		if err != nil {
			log.Printf("Data dump error: %s", err.Error())
		}
	}
}

// LambdaHandler is a lambda handler that takes in no input and returns an error
// See https://docs.aws.amazon.com/lambda/latest/dg/go-programming-model-handler-types.html
func LambdaHandler() error {
	log.Printf("Starting ingestion...")
	return ingest()
}
