package main

import (
	"flag"
	"os"
	"fmt"
	"net/http"
	"context"
	"text/tabwriter"
	"strings"
	"regexp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	awsLambda "github.com/aws/aws-sdk-go/service/lambda"
	"code.justin.tv/foundation/twitchclient"
	contacts "code.justin.tv/availability/MyFultonContactsServiceLambdaTwirp"
	lambdaTransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
)

const uidDesc = "Login User Id"
const eidDesc = "Amazon Employee Number"
const contactsServiceUrl = "https://contacts-api.internal.justin.tv"

// getContactsServiceClient is a helper function that returns an instance of the contacts service client
func getContactsServiceClient() (contacts.MyFultonContactsServiceService, error) {
	contactsConf := twitchclient.ClientConf{
		Host: "arn:aws:lambda:us-west-2:117703951204:function:MyFultonContactsServiceLambda-LambdaFunction-1FXZU65CZ6XXT",
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 75,
		},
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			return http.ErrUseLastResponse
		},
	}
	contactsTwirpClient := twitchclient.NewHTTPClient(contactsConf)
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		return nil, err
	}

	contactsTwirpConfig := sess.Config.WithHTTPClient(contactsTwirpClient)
	contactsTwirpLambdaClient := awsLambda.New(session.Must(session.NewSession(contactsTwirpConfig)))

	contactsService := contacts.NewMyFultonContactsServiceServiceJSONClient("",
		lambdaTransport.NewClient(contactsTwirpLambdaClient, contactsConf.Host))

	return contactsService, nil
}

func assertUserFlags(uid *string, eid *uint) bool {
	if *uid == "" && *eid == 0 {
		fmt.Println("Please use either -uid or -eid to specify the person of interest")
		return false
	}
	return true
}

func exitIfError(err error) {
	if err != nil {
		fmt.Printf("Error: %s", err.Error())
		fmt.Println()
		os.Exit(1)
	}
}

func printHeaders(tw *tabwriter.Writer, colNames []string) {
	headerLine := strings.Join(colNames, "\t")
	fmt.Fprintln(tw, headerLine)
	reg, _ := regexp.Compile("[a-zA-Z]")
	separatorLine := reg.ReplaceAllString(headerLine, "-")
	fmt.Fprintln(tw, separatorLine)
}

func printGeneralHelp(commands map[string]*flag.FlagSet) {
	fmt.Println("usage: contacts <command> [options]")
	fmt.Println()
	fmt.Println("available commands are:")
	for k, _ := range commands {
		fmt.Printf("* %s\n", k)
	}
	fmt.Println()
	fmt.Println("for help on an individual command: contacts <command> -h")
}

func main() {
	commands := make(map[string]*flag.FlagSet)

	getStructureCmd := flag.NewFlagSet("orgstructure", flag.ExitOnError)
	gsUserIdFlag := getStructureCmd.String("uid", "", uidDesc)
	gsEmpIdFlag := getStructureCmd.Uint("eid", 0, eidDesc)
	commands["orgstructure"] = getStructureCmd

	listReportsCmd := flag.NewFlagSet("reports", flag.ExitOnError)
	lrUserIdFlag := listReportsCmd.String("uid", "", uidDesc)
	lrEmpIdFlag := listReportsCmd.Uint("eid", 0, eidDesc)
	lrIncludeMgrFlag := listReportsCmd.Bool("includemgr", false, "Include Manager")
	lrDepthFlag := listReportsCmd.Uint("depth", 20, "Depth of Reports")
	commands["reports"] = listReportsCmd

	getManagerCmd := flag.NewFlagSet("manager", flag.ExitOnError)
	gmUserIdFlag := getManagerCmd.String("uid", "", uidDesc)
	gmEmpIdFlag := getManagerCmd.Uint("eid", 0, eidDesc)
	commands["manager"] = getManagerCmd

	listBizUnitsCmd := flag.NewFlagSet("bizunits", flag.ExitOnError)
	commands["bizunits"] = listBizUnitsCmd

	listOrgsCmd := flag.NewFlagSet("organizations", flag.ExitOnError)
	loBizUnitIdFlag := listOrgsCmd.Uint("buid", 0, "Business Unit ID")
	commands["organizations"] = listOrgsCmd

	listTeamsCmd := flag.NewFlagSet("teams", flag.ExitOnError)
	ltOrgIdFlag := listTeamsCmd.Uint("oid", 0, "Org ID")
	commands["teams"] = listTeamsCmd

	getPersonCmd := flag.NewFlagSet("person", flag.ExitOnError)
	gpUserIdFlag := getPersonCmd.String("uid", "", uidDesc)
	gpEmpIdFlag := getPersonCmd.Uint("eid", 0, eidDesc)
	commands["person"] = getPersonCmd

	if len(os.Args) == 1 {
		printGeneralHelp(commands)
		return
	}

	defaultCtx := context.Background()
	tw := tabwriter.NewWriter(os.Stdout, 0, 8, 1, '\t', 0)

	api, err := getContactsServiceClient()
	if err != nil {
		fmt.Fprintf(tw, "unable to get contacts service client due to: %s\n", err.Error())
		return
	}

	if commands[os.Args[1]] == nil {
		printGeneralHelp(commands)
		return
	}

	commands[os.Args[1]].Parse(os.Args[2:])

	if getStructureCmd.Parsed() {
		if !assertUserFlags(gsUserIdFlag, gsEmpIdFlag) {
			return
		}
		result, err := api.GetStructure(defaultCtx, &contacts.UserRequest{UserId: *gsUserIdFlag, EmployeeId: uint32(*gsEmpIdFlag)})
		exitIfError(err)

		fmt.Fprintf(tw, "Business Unit:\t%s\n", result.Bu.Name)
		fmt.Fprintf(tw, "Organization:\t%s\n", result.Org.Name)
		fmt.Fprintf(tw, "Team:\t%s\n", result.Team.Name)
		tw.Flush()
	}

	if listReportsCmd.Parsed() {
		if !assertUserFlags(lrUserIdFlag, lrEmpIdFlag) {
			return
		}
		result, err := api.ListReports(defaultCtx, &contacts.GetReportsRequest{UserId: *lrUserIdFlag, EmployeeId: uint32(*lrEmpIdFlag), IncludeManager: *lrIncludeMgrFlag, Depth: uint32(*lrDepthFlag)})
		exitIfError(err)
		printHeaders(tw, []string{"Name", "uid", "Title", "Team"})
		for _, e := range result.Persons {
			fmt.Fprintf(tw, "%s\t%s\t%s\t%s", e.PreferredName, e.UserId, e.Title, e.Team.Name)
			fmt.Fprintln(tw)
		}
		tw.Flush()
	}

	if getManagerCmd.Parsed() {
		if !assertUserFlags(gmUserIdFlag, gmEmpIdFlag) {
			return
		}
		mgr, err := api.GetManager(defaultCtx, &contacts.UserRequest{UserId: *gmUserIdFlag, EmployeeId: uint32(*gmEmpIdFlag)})
		exitIfError(err)

		fmt.Fprintf(tw, "Preferred Name:\t%s\n", mgr.Person.PreferredName)
		fmt.Fprintf(tw, "User ID:\t%s\n", mgr.Person.UserId)
		fmt.Fprintf(tw, "Employee Number:\t%d\n", mgr.Person.EmployeeNumber)
		fmt.Fprintf(tw, "Email:\t%s\n", mgr.Person.Email)
		fmt.Fprintf(tw, "Title:\t%s\n", mgr.Person.Title)
		tw.Flush()
	}

	if listBizUnitsCmd.Parsed() {
		result, err := api.ListBUs(defaultCtx, &contacts.BUsRequest{})
		exitIfError(err)
		printHeaders(tw, []string{"ID", "Name"})
		for _, e := range result.Bus {
			fmt.Fprintf(tw, "%d\t%s", e.Id, e.Name)
			fmt.Fprintln(tw)
		}
		tw.Flush()
	}

	if listOrgsCmd.Parsed() {
		if *loBizUnitIdFlag == 0 {
			fmt.Println("Please use -buid to specify the BU")
			return
		}
		result, err := api.ListOrgs(defaultCtx, &contacts.OrgsRequest{BuId: uint32(*loBizUnitIdFlag)})
		exitIfError(err)
		printHeaders(tw, []string{"ID", "Name"})
		for _, e := range result.Orgs {
			fmt.Fprintf(tw, "%d\t%s", e.Id, e.Name)
			fmt.Fprintln(tw)
		}
		tw.Flush()
	}

	if listTeamsCmd.Parsed() {
		if *ltOrgIdFlag == 0 {
			fmt.Println("Please use -oid to specify the Org")
			return
		}
		result, err := api.ListTeams(defaultCtx, &contacts.TeamsRequest{OrgId: uint32(*ltOrgIdFlag)})
		exitIfError(err)
		printHeaders(tw, []string{"ID", "Name"})
		for _, e := range result.Teams {
			fmt.Fprintf(tw, "%d\t%s", e.Id, e.Name)
			fmt.Fprintln(tw)
		}
		tw.Flush()
	}

	if getPersonCmd.Parsed() {
		if !assertUserFlags(gpUserIdFlag, gpEmpIdFlag) {
			return
		}
		p, err := api.GetPerson(defaultCtx, &contacts.UserRequest{UserId: *gpUserIdFlag, EmployeeId: uint32(*gpEmpIdFlag)})
		exitIfError(err)

		structRsp, err := api.GetStructure(defaultCtx, &contacts.UserRequest{UserId: *gpUserIdFlag, EmployeeId: uint32(*gpEmpIdFlag)})
		exitIfError(err)
		mgrRsp, err := api.GetManager(defaultCtx, &contacts.UserRequest{UserId: *gpUserIdFlag, EmployeeId: uint32(*gpEmpIdFlag)})
		exitIfError(err)

		fmt.Fprintf(tw, "Preferred Name:\t%s\n", p.Person.PreferredName)
		fmt.Fprintf(tw, "User ID:\t%s\n", p.Person.UserId)
		fmt.Fprintf(tw, "Employee Number:\t%d\n", p.Person.EmployeeNumber)
		fmt.Fprintf(tw, "Email:\t%s\n", p.Person.Email)
		fmt.Fprintf(tw, "Title:\t%s\n", p.Person.Title)
		fmt.Fprintf(tw, "Business Unit:\t%s\n", structRsp.Bu.Name)
		fmt.Fprintf(tw, "Organization:\t%s\n", structRsp.Org.Name)
		fmt.Fprintf(tw, "Team:\t%s\n", structRsp.Team.Name)
		fmt.Fprintf(tw, "Manager Name:\t%s\n", mgrRsp.Person.PreferredName)
		fmt.Fprintf(tw, "Manager Title:\t%s\n", mgrRsp.Person.Title)
		tw.Flush()
	}
}
