package contactsserver

import (
	"os"
	"time"
	"log"
	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"strings"
)

const contactsServiceSecretPrefix = "qa-eng/contacts-service/"

func SetupSandstorm() (*manager.Manager, error) {
	env := os.Getenv("ENVIRONMENT")
	roleArn := "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/qe-" + env + "-contacts-service"
	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, err
	}
	stsClient := sts.New(awsSession)
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      roleArn,
		Client:       stsClient,
	}
	omlCredentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(omlCredentials)
	sandstorm := manager.New(manager.Config{
		AWSConfig: awsConfig,
		ServiceName: "automation-webhooks",
	})
	sandstorm.ListenForUpdates()
	return sandstorm, nil
}

func GetSecrets(sandstorm *manager.Manager, secretKeyName string) string {
	raw, err := sandstorm.Get(secretKeyName)
	if err != nil {
		log.Fatal(err.Error())
		return ""
	}
	return string(raw.Plaintext)
}

func GetEnvVarOrSecrets(sandstorm *manager.Manager, secretKeyName string) string {
	value := os.Getenv(strings.ToUpper(secretKeyName)) // by convention, env vars are uppercase, sandstorm key names are lowercase
	if value != "" {
		return value
	}

	if sandstorm != nil {
		env := os.Getenv("ENVIRONMENT")
		fullSecretsKey := contactsServiceSecretPrefix + env + "/" + secretKeyName
		value = GetSecrets(sandstorm, fullSecretsKey)
		log.Printf("retrieved %s from sandstorm: %s", fullSecretsKey, strings.Repeat("*", len(value)))
	}

	return value
}