package contactsserver

import (
	d "code.justin.tv/qe/contacts-service/model/db"
	"code.justin.tv/qe/contacts-service/rpc/contacts"
	"context"
	"github.com/jinzhu/copier"
	"github.com/jinzhu/gorm"
	"github.com/twitchtv/twirp"
	"fmt"
	"strings"
)

const (
	personColumns = "id,user_id,employee_number,email,first_name,last_name,preferred_name,team_id,manager_id,title,twitch_amzn_uid,employee_type,location_id,is_active,is_manager"
	maxDepth = 20
)

// reference: http://www.postgresqltutorial.com/postgresql-recursive-query/
var indirectReportsIncludeManager = fmt.Sprintf(`WITH RECURSIVE reports(depth,%s) AS (
  SELECT 0,%s
  FROM PERSON
  WHERE user_id = ?
  UNION
  SELECT (r.depth+1),p.%s
  FROM person p, reports r
    WHERE r.id = p.manager_id
    AND r.depth < ?
) SELECT %s
  FROM reports`, personColumns, personColumns, strings.Replace(personColumns, ",", ",p.", -1), personColumns)

var indirectReportsExcludeManager = indirectReportsIncludeManager + "\n" +
  "WHERE user_id != ?"


// Server implements the Contacts service
type Server struct {
	DB *gorm.DB
}

func (s *Server) getPerson(request *contacts.UserRequest) (*d.Person, error) {
	userIdProvided := len(request.UserId) > 0
	employeeIdProvided := request.EmployeeId > 0
	if !(userIdProvided != employeeIdProvided) { // not xor
		return nil, twirp.InvalidArgumentError("user_id or employee_id", "exactly one must be defined")
	}
	person := d.Person{}
	if employeeIdProvided {
		s.DB.First(&person, d.Person{EmployeeNumber: request.EmployeeId})
	} else {
		s.DB.First(&person, d.Person{UserId: request.UserId})
	}

	if person.ID == 0 {
		return nil, twirp.InvalidArgumentError("user_id",
			fmt.Sprintf("Person not found (user_id %s, employee_id %d)", request.UserId, request.EmployeeId))
	}
	return &person, nil
}

func (s *Server) GetStructure(ctx context.Context, request *contacts.UserRequest) (*contacts.StructureResponse, error) {
	person, err := s.getPerson(request)
	if err != nil {
		return nil, err
	}

	// db models
	team := d.Team{}
	s.DB.First(&team, person.TeamId)
	if team.ID == 0 {
		return nil, twirp.InvalidArgumentError("user_id", "No team linked to user_id")
	}

	org := d.Org{}
	s.DB.First(&org, team.OrgId)
	if org.ID == 0 {
		return nil, twirp.InvalidArgumentError("user_id", "No org linked to user_id")
	}

	bu := d.BU{}
	s.DB.First(&bu, org.BUId)
	if bu.ID == 0 {
		return nil, twirp.InvalidArgumentError("user_id", "No bu linked to user_id")
	}

	// response structures -- copy over the fields from db models
	protoTeam := contacts.ProtoTeam{}
	protoOrg := contacts.ProtoOrg{}
	protoBu := contacts.ProtoBU{}
	copier.Copy(&protoTeam, team)
	protoTeam.Id = uint32(team.ID)
	copier.Copy(&protoOrg, org)
	copier.Copy(&protoBu, bu)
	responseStruct := contacts.StructureResponse{Team: &protoTeam, Org: &protoOrg, Bu: &protoBu}

	return &responseStruct, nil
}

func (s *Server) ListReports(ctx context.Context, request *contacts.GetReportsRequest) (*contacts.ReportsResponse, error) {
	if request.Depth <= 0 || request.Depth > maxDepth {
		return nil, twirp.InvalidArgumentError("depth", "depth must be between 1 and 20 (inclusive)")
	}

	userRequest := &contacts.UserRequest{UserId: request.UserId, EmployeeId:request.EmployeeId}
	person, err := s.getPerson(userRequest)
	if err != nil {
		return nil, err // don't proceed with invalid user
	}

	// db models
	var reports []*d.Person
	depth := request.Depth
	if request.IncludeManager {
		s.DB.Raw(indirectReportsIncludeManager, person.UserId, depth).Scan(&reports)
	} else {
		s.DB.Raw(indirectReportsExcludeManager, person.UserId, depth, person.UserId).Scan(&reports)
	}

	// response structures -- copy over the fields from db models
	var protoReports []*contacts.ProtoPerson
	copier.Copy(&protoReports, reports)
	teamCache := make(map[uint]d.Team)
	locCache := make(map[uint]d.Location)
	for idx, _ := range reports {
		protoReports[idx].Id = uint32(reports[idx].ID)

		var protoTeam contacts.ProtoTeam
		team := d.Team{}
		if val, ok := teamCache[reports[idx].TeamId]; ok {
			team = val
		} else {
			s.DB.First(&team, reports[idx].TeamId)
			teamCache[reports[idx].TeamId] = team
		}
		copier.Copy(&protoTeam, &team)
		protoReports[idx].Team = &protoTeam

		var protoLoc contacts.ProtoLocation
		loc := d.Location{}
		if val, ok := locCache[reports[idx].LocationId]; ok {
			loc = val
		} else {
			s.DB.First(&loc, reports[idx].LocationId)
			locCache[reports[idx].LocationId] = loc
		}
		copier.Copy(&protoLoc, &loc)
		protoReports[idx].Location = &protoLoc
	}
	responseStruct := contacts.ReportsResponse{Persons: protoReports}

	return &responseStruct, nil
}

func (s *Server) GetManager(ctx context.Context, request *contacts.UserRequest) (*contacts.ManagerResponse, error) {
	person, err := s.getPerson(request)
	if err != nil {
		return nil, err
	}

	// db model
	manager := d.Person{}
	s.DB.First(&manager, person.ManagerId)

	// response structure -- copy over the fields from db model
	protoManager := contacts.ProtoPerson{}
	copier.Copy(&protoManager, manager)
	protoManager.Id = uint32(manager.ID)

	var protoTeam contacts.ProtoTeam
	var team d.Team
	s.DB.First(&team, manager.TeamId)
	copier.Copy(&protoTeam, &team)
	protoManager.Team = &protoTeam

	var protoLoc contacts.ProtoLocation
	var loc d.Location
	s.DB.First(&loc, manager.LocationId)
	copier.Copy(&protoLoc, &loc)
	protoManager.Location = &protoLoc

	responseStruct := contacts.ManagerResponse{Person: &protoManager}

	return &responseStruct, nil
}

func (s *Server) ListBUs(context.Context, *contacts.BUsRequest) (*contacts.BUsResponse, error) {
	var bus []*d.BU
	s.DB.Find(&bus)
	var protoBUs []*contacts.ProtoBU
	copier.Copy(&protoBUs, bus)
	for idx, _ := range bus {
		protoBUs[idx].Id = uint32(bus[idx].ID)
	}
	responseStruct := contacts.BUsResponse{Bus: protoBUs}

	return &responseStruct, nil
}

func (s *Server) ListOrgs(ctx context.Context, request *contacts.OrgsRequest) (*contacts.OrgsResponse, error) {
	var orgs []*d.Org
	if request.BuId == 0 {
		s.DB.Find(&orgs)
	} else {
		s.DB.Where(d.Org{BUId: uint(request.BuId)}).Find(&orgs)
	}
	var protoOrgs []*contacts.ProtoOrg
	copier.Copy(&protoOrgs, orgs)
	for idx, _ := range orgs {
		protoOrgs[idx].Id = uint32(orgs[idx].ID)
	}
	responseStruct := contacts.OrgsResponse{Orgs: protoOrgs}

	return &responseStruct, nil
}

func (s *Server) ListTeams(ctx context.Context, request *contacts.TeamsRequest) (*contacts.TeamsResponse, error) {
	var teams []*d.Team
	if request.OrgId == 0 {
		s.DB.Find(&teams)
	} else {
		s.DB.Where(d.Team{OrgId: uint(request.OrgId)}).Find(&teams)
	}
	var protoTeams []*contacts.ProtoTeam
	copier.Copy(&protoTeams, teams)
	for idx, _ := range teams {
		protoTeams[idx].Id = uint32(teams[idx].ID)
	}
	responseStruct := contacts.TeamsResponse{Teams: protoTeams}

	return &responseStruct, nil
}

func (s *Server) GetPerson(ctx context.Context, request *contacts.UserRequest) (*contacts.PersonResponse, error) {
	person, err := s.getPerson(request)
	if err != nil {
		return nil, err
	}

	// response structure -- copy over the fields from db model
	protoPerson := contacts.ProtoPerson{}
	copier.Copy(&protoPerson, person)

	var protoTeam contacts.ProtoTeam
	var team d.Team
	s.DB.First(&team, person.TeamId)
	copier.Copy(&protoTeam, &team)
	protoPerson.Team = &protoTeam

	var protoLoc contacts.ProtoLocation
	var loc d.Location
	s.DB.First(&loc, person.LocationId)
	copier.Copy(&protoLoc, &loc)
	protoPerson.Location = &protoLoc

	responseStruct := contacts.PersonResponse{Person: &protoPerson}

	return &responseStruct, nil
}

func (s *Server) ListPersons(ctx context.Context, request *contacts.PersonsRequest) (*contacts.PersonsResponse, error) {
	var persons []*d.Person
	s.DB.Find(&persons)

	var protoPersons []*contacts.ProtoPerson
	copier.Copy(&protoPersons, persons)
	managerCache := make(map[uint]d.Person)
	teamCache := make(map[uint]d.Team)
	orgCache := make(map[uint]d.Org)
	buCache := make(map[uint]d.BU)
	locCache := make(map[uint]d.Location)
	for idx, _ := range persons {
		protoPersons[idx].Id = uint32(persons[idx].ID)
		var protoTeam contacts.ProtoTeam
		team := d.Team{}
		if val, ok := teamCache[persons[idx].TeamId]; ok {
			team = val
		} else {
			s.DB.First(&team, persons[idx].TeamId)
			teamCache[persons[idx].TeamId] = team
		}
		copier.Copy(&protoTeam, &team)
		protoPersons[idx].Team = &protoTeam
		protoPersons[idx].Team.Id = uint32(team.ID)

		var protoOrg contacts.ProtoOrg
		org := d.Org{}
		if val, ok := orgCache[team.OrgId]; ok {
			org = val
		} else {
			s.DB.First(&org, team.OrgId)
			orgCache[team.OrgId] = org
		}
		copier.Copy(&protoOrg, &org)
		protoPersons[idx].Team.Org = &protoOrg
		protoPersons[idx].Team.Org.Id = uint32(org.ID)

		var protoBu contacts.ProtoBU
		bu := d.BU{}
		if val, ok := buCache[org.BUId]; ok {
			bu = val
		} else {
			s.DB.First(&bu, org.BUId)
			buCache[org.BUId] = bu
		}
		copier.Copy(&protoBu, &bu)
		protoPersons[idx].Team.Org.Bu = &protoBu
		protoPersons[idx].Team.Org.Bu.Id = uint32(bu.ID)


		var protoLoc contacts.ProtoLocation
		loc := d.Location{}
		if val, ok := locCache[persons[idx].LocationId]; ok {
			loc = val
		} else {
			s.DB.First(&loc, persons[idx].LocationId)
			locCache[persons[idx].LocationId] = loc
		}
		copier.Copy(&protoLoc, &loc)
		protoPersons[idx].Location = &protoLoc

		var protoManager contacts.ProtoPerson
		manager := d.Person{}
		if val, ok := managerCache[persons[idx].ManagerId]; ok {
			manager = val
		} else {
			s.DB.First(&manager, persons[idx].ManagerId)
			managerCache[persons[idx].ManagerId] = manager
		}
		copier.Copy(&protoManager, &manager)
		protoPersons[idx].Manager = &protoManager

	}
	responseStruct := contacts.PersonsResponse{Persons: protoPersons}
	return &responseStruct, nil
}
