package contactsserver

import (
	"code.justin.tv/qe/contacts-service/rpc/contacts"
	"context"
	_ "github.com/jinzhu/gorm/dialects/postgres"
	"net/http"
	"testing"
	"fmt"
	"strings"
	"encoding/json"
	"time"
	"os"
	"github.com/jinzhu/gorm"
	"log"
)

// Mock code cov server
type MockCodeCovServer struct {}

// dummy response for testing
func (s *MockCodeCovServer) getCodeCov() [5]map[string]string {
	return [5]map[string]string{
		{"repo": "video/player-core", "owner": "purushen", "coverage": "50"},
		{"repo": "video/player-ui", "owner": "aisaac", "coverage": "60"},
		{"repo": "qe/contacts-service", "owner": "lukemng", "coverage": "70"},
		{"repo": "qe/heimdall", "owner": "dylan", "coverage": "80"},
		{"repo": "chat/tmi", "owner": "lukemng", "coverage": "60"},
	}
}

type ServiceCatalogMessage struct {
	Data struct {
		Services []struct {
			Id     string `json:"id"`
			Name   string `json:"name"`
			TeamId string `json:"team_id"`
			PrimaryOwner struct {
				Cn  string `json:"cn"`
				Uid string `json:"uid"`
			} `json:"primary_owner"`
		} `json:"services"`
	} `json:"data"`
}

// Lightweight service catalog client
type ServiceCatalog struct {}

func (s *ServiceCatalog) getServices() ServiceCatalogMessage {
	payload := strings.NewReader(`{"operationName":"FetchAllServices","variables":{},"query":"query FetchAllServices {\n  services {\n    ...serviceShallowData\n    }\n}\n\nfragment serviceShallowData on Service {\n  id\n  name\n  team_id\n  primary_owner{\n cn\n uid\n}\n \n    }\n"}`)
	serviceCatalogResponse := ServiceCatalogMessage{}
	for i := 0; i < 3; i++ {
		res, err := http.Post("https://status.internal.justin.tv/api/v2/query", "application/x-www-form-urlencoded", payload)
		if err != nil {
			println("could not retrieve list of services from service catalog")
		}
		json.NewDecoder(res.Body).Decode(&serviceCatalogResponse)
		if len(serviceCatalogResponse.Data.Services) > 0 {
			break
		}
		time.Sleep(500 * time.Millisecond)
	}
	return serviceCatalogResponse
}

// helper to setup server
func NewContactsTwirpServer() contacts.TwirpServer {
	dbUrl := os.Getenv("CONTACTS_DB_URL")
	db, err := gorm.Open("postgres", dbUrl)
	if err != nil {
		log.Fatalf("Could not connect to database at %s due to: %s", dbUrl, err.Error())
	}
	db.SingularTable(true)

	server := &Server{DB: db}
	return contacts.NewContactsServer(server, nil)
}

// init() performs global startup initialization - http://cs-guy.com/blog/2015/01/test-main/
func init() {
	twirpHandler := NewContactsTwirpServer()
	go http.ListenAndServe(":" + localPort, twirpHandler)
	time.Sleep(100 * time.Millisecond)
	log.Println("started local server")
}
const localPort = "8888"
const contactsServiceUrl = "http://localhost:" + localPort
//const contactsServiceUrl = "https://contacts-api-staging.internal.justin.tv"
//const contactsServiceUrl = "https://contacts-api.internal.justin.tv"

func TestGetStructure(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "lukemng"
	result, err := api.GetStructure(context.Background(), &contacts.UserRequest{UserId: userId})
	if err != nil {
		t.Fatalf("fail due to %s", err.Error())
	}
	t.Logf("%s's org structure: BU=%s, Org=%s, Team=%s", userId, result.Bu.Name, result.Org.Name, result.Team.Name)
}

func TestListReports(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "tylerobb"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth:1})
	if err != nil {
		t.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Persons {
		t.Logf("%s - %s's report = %s (location: %s)", e.PreferredName, userId, e.UserId, e.Location.BuildingName)
		if e.Location == nil {
			t.Fatalf("location is nil")
		}
		if e.Team == nil {
			t.Fatalf("team is nil")
		}
	}
	expectedBallpark := 15
	actualReportsCount := len(result.Persons)
	if  actualReportsCount > expectedBallpark {
		t.Fatalf("failed. expected fewer than than %d entries. actual count=%d", expectedBallpark, actualReportsCount)
	}
}

func TestListReportsIndirects(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "tylerobb"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 20})
	if err != nil {
		t.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Persons {
		t.Logf("%s = %s's report = %s (location: %s)", e.PreferredName, userId, e.UserId, e.Location.BuildingName)
	}
	expectedBallpark := 30
	actualReportsCount := len(result.Persons)
	if  actualReportsCount <= expectedBallpark {
		t.Fatalf("failed. expected more than %d entries. actual count=%d", expectedBallpark, actualReportsCount)
	}
}

func TestListReportsIndirectsAndLimitedDepth(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "tylerobb"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 2})
	if err != nil {
		t.Fatalf("fail due to %s", err.Error())
	}
	for _, e := range result.Persons {
		t.Logf("%s = %s's report = %s (location: %s)", e.PreferredName, userId, e.UserId, e.Location.BuildingName)
	}
	expectedBallpark := 30
	actualReportsCount := len(result.Persons)
	if  actualReportsCount > expectedBallpark {
		t.Fatalf("failed. expected fewer than than %d entries. actual count=%d", expectedBallpark, actualReportsCount)
	}
}

func TestListReportsInvalidDepth1(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "tylerobb"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 0})
	if err == nil {
		t.Fatalf("failed because expected error not found")
	}
	if result != nil && len(result.Persons) > 0 {
		t.Fatalf("failed because entries were returned while expecting none")
	}
	t.Logf("found expected err: %s", err.Error())
}

func TestListReportsInvalidDepth2(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "tylerobb"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 21})
	if err == nil {
		t.Fatalf("failed because expected error not found")
	}
	if result != nil && len(result.Persons) > 0 {
		t.Fatalf("failed because entries were returned while expecting none")
	}
	t.Logf("found expected err: %s", err.Error())
}

func TestListReportsInvalidUserId(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	userId := "asdhfasdhfjkahsfjkahlsk"
	result, err := api.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: userId, Depth: 10})
	if err == nil {
		t.Fatalf("failed because expected error not found")
	}
	if result != nil && len(result.Persons) > 0 {
		t.Fatalf("failed because entries were returned while expecting none")
	}
	t.Logf("found expected err: %s", err.Error())
}

func TestListPersons(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	ctx := context.Background()
	persons, err := api.ListPersons(ctx, &contacts.PersonsRequest{})
	if err != nil {
		t.Fatalf("failed due to : %s", err.Error())
	}
	if len(persons.Persons) == 0 {
		t.Fatalf("failed due to len(persons.Persons) == 0")
	}
	for _, person := range persons.Persons {
		t.Logf("%v", person)
		if person.Team.Id == 0 {
			t.Fatalf("failed because team id is 0")
		}
		if person.Team.Org == nil {
			t.Fatalf("failed because org is nil")
		}
		if person.Team.Org.Bu == nil {
			t.Fatalf("failed because bu is nil")
		}
	}
}

func DISABLED_TestPerUserReportingPathToTop(t *testing.T) {
	api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
	ctx := context.Background()
	persons, err := api.ListPersons(ctx, &contacts.PersonsRequest{})
	if err != nil {
		t.Fatalf("failed due to : %s", err.Error())
	}
	userId2MgrUserId := make(map[string]string)
	ldapId2AmznId := make(map[string]string)
	for _, person := range persons.Persons {
		reachedTop := false

		pUid := person.UserId
		amznUid := person.TwitchAmznUid
		ldapId2AmznId[pUid] = amznUid

		var path []string
		path = append(path, amznUid)

		for !reachedTop {
			if cachedUid, ok := userId2MgrUserId[pUid]; ok {
				pUid = cachedUid
				amznUid = ldapId2AmznId[pUid]
			} else {
				p, err := api.GetManager(ctx, &contacts.UserRequest{UserId: pUid})
				if err != nil {
					t.Fatalf("failed due to : %s", err.Error())
				}
				userId2MgrUserId[pUid] = p.Person.UserId
				pUid = p.Person.UserId
				amznUid = p.Person.TwitchAmznUid
				ldapId2AmznId[pUid] = amznUid
			}

			if pUid == "" {
				reachedTop = true
				break
			}
			path = append(path, amznUid)
		}
		fmt.Println(strings.Join(path, ","))
	}
}
