# cw_es_shipper
This Go application is a Lambda Function that will take a Cloudwatch Logs Event and ship it to an Elasticsearch Cluster.

## Terraform Usage
A Terraform v0.11 Module is provided for usage of this application.

Example `main.tf`:
```
# Create Log Groups for Testing
resource "aws_cloudwatch_log_group" "app_logs_1" {
  name = "app_logs_1"
}
resource "aws_cloudwatch_log_group" "app_logs_2" {
  name = "app_logs_2"
}

# Create a Lambda to ship Cloudwatch Logs to Elasticsearch
module "cw_es_shipper" {
  source = "git::git+ssh://git@git.xarth.tv/qe/cw_es_shipper//terraform/modules/lambda?ref=v1.0.0"
  elasticsearch_endpoint_url = "https://es-domain-vjs47bt4a4c2nfjceo2gibltmi.us-west-2.es.amazonaws.com"
  elasticsearch_arn          = "arn:aws:es:us-west-2:ACCOUNT_ID:domain/ES_DOMAIN_NAME"
}

# Add a subscription to Log Group "App Logs 1" to ship to Elasticsearch
module "subscription" {
  source = "git::git+ssh://git@git.xarth.tv/qe/cw_es_shipper//terraform/modules/cw_logs_subscription?ref=v1.0.0"
  cloudwatch_log_group_name = "${aws_cloudwatch_log_group.app_logs_1.name}"
  lambda_name = "${module.cw_es_shipper.aws_lambda_function_name}"
}

# Add a subscription to Log Group "App Logs 2" to ship to Elasticsearch
module "subscription2" {
  source ="git::git+ssh://git@git.xarth.tv/qe/cw_es_shipper//terraform/modules/cw_logs_subscription?ref=v1.0.0"
  cloudwatch_log_group_name = "${aws_cloudwatch_log_group.app_logs_2.name}"
  lambda_name = "${module.cw_es_shipper.aws_lambda_function_name}"
}

# Output the IAM Role ARN for easy update of Elasticsearch Access Policy
output "lambda_iam_role_arn" {
  value = "${module.cw_es_shipper.lambda_iam_role_arn}"
}
```

Then run `terraform init` and `terraform apply`

After the run completes, an IAM Role will be outputted, such as:
```
Apply complete! Resources: 14 added, 0 changed, 0 destroyed.

Outputs:

lambda_iam_role_arn = arn:aws:iam::011102325390:role/cloudwatch_elasticsearch_shipper_3bA6tk9e
```

Then, give the Lambda access via the Elasticsearch Service Access Policy by appending the Principal AWS ARN:
```
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "arn:aws:iam::011102325390:role/cloudwatch_elasticsearch_shipper_3bA6tk9e",
          "another_arn_this_is_just_an_example_you_can_remove"
        ]
      },
      "Action": "es:ESHttpPost",
      "Resource": "arn:aws:es:us-west-2:000368311237:domain/dylan-testing-es/*"
    }
  ]
}
```

Ensure the Principal AWS ARN matches the IAM ARN printed out, and that the Resource is your Elasticsearch Service's ARN

## Expectations

### Input
It expects Cloudwatch Logs to provide an event as a Base64 encoded GZip Archive data message, as described in the [documentation](https://docs.aws.amazon.com/lambda/latest/dg/services-cloudwatchlogs.html). Example:
```
{
  "awslogs": {
    "data": "ewogICAgIm1lc3NhZ2VUeXBlIjogIkRBVEFfTUVTU0FHRSIsCiAgICAib3duZXIiOiAiMTIzNDU2Nzg5MDEyIiwKICAgICJsb2dHcm91cCI6I..."
  }
}
```

### Parsing
The Log Event "message" is expected to be in JSON Format, and will set each JSON Key as a field in Elasticsearch.
For example: `"message": "{\"level\": \"info\", \"message\": \"this is a message from the application\"}"`

will be stored in Elasticsearch with two fields:
`level: info` and `message: this is a message from the application`

This is dynamic, allowing for applications with different logging fields to report all their fields.

If a user does **not** provide a JSON encoded body for the message, the body will be sent as `@message` to Elasticsearch

### Index
The Elasticsearch Index will be based on the date when the lambda runs. On January 2nd, 2006, it will be in the format of:
`cwl-2006.01.02`

This application uses AWS SigV4 when using the Elasticsearch Bulk operation, so your Lambda will require Post permissions to your Elasticsearch Instance

## Environment Variable
| Environment Variable | Default | Description | Example |
| -------------------- | ------- | ------------- |------ |
| ES_ENDPOINT | REQUIRED | The Elasticsearch Cluster endpoint to connect to | `https://search-dylan-es-test-hsz3h7rt5nklhg7ga6gtgqdtuq.us-west-2.es.amazonaws.com`
| AWS_REGION | us-west-2 | The region the Elasticsearch Cluster is running in | `us-west-2`

## Building
This application uses Go Modules to build dependencies

`GOOS=linux go build ./...`

## Running Locally
Currently not supported

## Testing
`go test ./...`
