package main

import (
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"time"
)

// Event sent to Lambda from Cloudwatch Logs
// Reference: https://docs.aws.amazon.com/lambda/latest/dg//services-cloudwatchlogs.html
type Event struct {
	Logs struct  {
		Data string `json:"data"` // Data field is a Base64 encoded ZIP archive
	} `json:"awslogs"`
}

// Amazon CloudWatch Logs Message Data (decoded)
// Reference: https://docs.aws.amazon.com/lambda/latest/dg//services-cloudwatchlogs.html
type CloudwatchLogsMessageData struct {
	MessageType         string     `json:"messageType"`
	Owner               string     `json:"owner"`
	LogGroup            string     `json:"logGroup"`
	LogStream           string     `json:"logStream"`
	SubscriptionFilters []string   `json:"subscriptionFilters"`
	LogEvents           []LogEvent `json:"logEvents"`
}

// Amazon Cloudwatch Log Event (decoded)
// Reference: https://docs.aws.amazon.com/lambda/latest/dg//services-cloudwatchlogs.html
type LogEvent struct {
	ID        string `json:"id"`
	Timestamp int64  `json:"timestamp"`
	Message   string `json:"message"`
}

// Returns the Index Format to use for Elasticsearch, in the format of cwl-2006.01.02
// It will indicate the day the log was generated from the current time
// Provide currentTime as time.Now() or a custom time for reasons like testing
func GetEsIndex(currentTime int64) string {
	timestamp := time.Unix(currentTime / 1000, 0).Format("2006.01.02")
	return fmt.Sprintf("cwl-%s", timestamp)
}

/*
Takes a LogEvent and converts it to a body that can be sent to Elasticsearch's Bulk API. Example:
{ "index" : { "_index" : "cwl-2006.01.02", "_id" : "1" } }
{ "field1" : "value1" }
*/
func (logEvent *LogEvent) ToElasticsearchBulkBody(payload *CloudwatchLogsMessageData) ([]byte, error) {
	line1 := ESIndex{ Index: ESIndexData{
		Index: GetEsIndex(logEvent.Timestamp),
		Type:  "cwl",
		ID:    logEvent.ID,
	} }

	indexData, err := json.Marshal(line1)
	if err != nil { return nil, err }

	// Create a map for all the json fields within the log message
	logMessageData := make(map[string]string)

	// Parse the json within the log event message
	err = json.Unmarshal([]byte(logEvent.Message), &logMessageData)
	if err != nil {
		// Silently ignore. The raw message will still be sent to Elasticsearch
		log.Printf("[WARN] The input message was not able to be parsed. Ensure it's in json format. The error: %v\nThe message:\n%s", err, logEvent.Message)
	}

	// Append additional metadata
	logMessageData["@id"]         = logEvent.ID
	logMessageData["@timestamp"]  = logEvent.GetParsedTimestamp()
	logMessageData["@message"]    = logEvent.Message
	logMessageData["@owner"]      = payload.Owner
	logMessageData["@log_group"]  = payload.LogGroup
	logMessageData["@log_stream"] = payload.LogStream

	// Marshal it back to json
	messageData, err := json.Marshal(logMessageData)
	if err != nil { return nil, err }

	// Return a string with the index data on first line, and message data on other
	// this is required format by the ES bulk operation
	return []byte(fmt.Sprintf("%s\n%s\n", indexData, messageData)), nil
}

// Convert Time Milliseconds to Seconds
// CW Gives us: "The time the event occurred, expressed as the number of milliseconds after Jan 1, 1970 00:00:00 UTC."
// https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_InputLogEvent.html
func (logEvent *LogEvent) GetParsedTimestamp() string {
	parsedTimestamp := time.Unix(logEvent.Timestamp / 1000, 0)
	return parsedTimestamp.Format(time.RFC3339)
}

/*
Takes all LogEvents associated with the Cloudwatch Log event and converts it to a body that can be sent to Elasticsearch's Bulk API. Example:
{ "index" : { "_index" : "cwl-2006.01.02", "_id" : "1" } }
{ "field1" : "value1" }

{ "index" : { "_index" : "cwl-2006.01.02", "_id" : "2" } }
{ "field2" : "value2" }
 */
func (cwData *CloudwatchLogsMessageData) ToElasticsearchBulkBody() ([]byte, error) {
	var fullBody string
	errorsFound := false

	// Loop through each LogEvent and convert that to the BulkBody format
	for _, logEvent := range cwData.LogEvents {
		body, err := logEvent.ToElasticsearchBulkBody(cwData)
		if err != nil {
			log.Printf("[WARN] Encountered error while parsing log event, so going to skip. Err: %v", err)
			errorsFound = true
			continue
		}

		// Append it to the fullBody, adding a newline at the end (required by Bulk API)
		fullBody = fmt.Sprintf("%s%s\n", fullBody, body)
	}

	// Return back any errors found
	if errorsFound {
		return []byte(fullBody), errors.New("encountered an error building bulk body")
	} else {
		return []byte(fullBody), nil
	}
}

// Will take data from the Amazon Cloudwatch (Base 64 Encoded and gziped) Logs and decode it
func (event *Event) DecodeData() ([]byte, error) {
	b64Data, err := base64.StdEncoding.DecodeString(event.Logs.Data)
	if err != nil { return nil, err }

	r, err := gzip.NewReader(bytes.NewReader(b64Data))
	if err != nil { return nil, err }

	result, err := ioutil.ReadAll(r)
	if err != nil { return nil, err }

	return result, nil
}
