package main

import (
	"encoding/json"
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"strings"
	"testing"
	"time"
)

func TestEvent_DecodeData(t *testing.T) {
	t.Run("properly decodes", func (t *testing.T) {
		// this is a real example from manually testing with Cloudwatch Logs + Lambda - we got this from CWL
		b64GZippedData := "H4sIAAAAAAAAADVQy26DMBD8lcjnUBsb25BbpKa5tJeGU0tVGdgQKwQj2yGqovx7F6rKe/GMdh57JxcIwXRQ/oxANuR5W26/33aHw3a/I2vibgN4hCXPhOAsVzrTCPeu23t3HZGh5hYo9CZE29RghhBNf6avruvAJzBMiaCT8RQ3KNRJ65ozeNq4IRqL0mEGm6v3MMTEjCMNsXXXCN4/4caf0yF6MBe0sglrWlC11EfgLZNCzlnCtQ6Nt2O0bnixPe4GsvkkWOFkxxHTfy0quwktZuZObItiQnKZpzJjrBBpoYTOcpFrlacFVhSCccU515IXLOdS61QpXWT4R8do8WbRXLB+ihpM4FNpztf/t0T5e0UO4CfbQEU2Fdl7267el2YVWVekhwn6hbHD0S3QJXQLUJ5sWOFE9Fhh8IWcLReWM84SxhOmSi42TOJ8VORBHl+PXx8/tVfLAQAA"

		decodedData := "{\"messageType\":\"DATA_MESSAGE\",\"owner\":\"524332086747\"," +
			"\"logGroup\":\"/aws/elasticbeanstalk/Logger-env-3/var/log/eb-docker/containers/eb-current-app/stdouterr.log\"," +
			"\"logStream\":\"i-0cde6b57fe2d05357\",\"subscriptionFilters\":[\"ESShipper\"]," +
			"\"logEvents\":[{\"id\":\"35258154009319637483876819747330262227529082577166794752\"," +
			"\"timestamp\":1581030306182,\"message\":\"{\\\"Service\\\":\\\"Grid Router\\\"," +
			"\\\"level\\\":\\\"info\\\",\\\"msg\\\":\\\"This is test log\\\",\\\"time\\\":\\\"2020-02-06T23:05:05Z\\\"}\"}]}"

		e := Event{
			Logs: struct{ Data string `json:"data"` }(struct{ Data string }{Data: b64GZippedData}),
		}

		result, err := e.DecodeData()
		assert.NoError(t, err)
		assert.Equal(t, decodedData, string(result))
	})

	t.Run("returns an error for bad base64", func (t *testing.T) {
		fakeBase64 := "hello world"
		e := Event{
			Logs: struct{ Data string `json:"data"` }(struct{ Data string }{Data: fakeBase64}),
		}

		result, err := e.DecodeData()
		assert.Error(t, err)
		assert.True(t, strings.Contains(err.Error(), "illegal base64 data"), err.Error() + " did not include the substring")
		assert.Nil(t, result)
	})

	t.Run("returns an error for bad gzip", func (t *testing.T) {
		b64NoGZip := "aGVsbG8gd29ybGQ=" // g64 for hello world
		e := Event{
			Logs: struct{ Data string `json:"data"` }(struct{ Data string }{Data: b64NoGZip}),
		}

		result, err := e.DecodeData()
		assert.Error(t, err)
		assert.EqualError(t, err, "gzip: invalid header")
		assert.Nil(t, result)
	})
}

func TestCloudwatchLogsMessageData_ToElasticsearchBulkBody(t *testing.T) {
	t.Run("properly encoded", func (t *testing.T) {
		currentTime, err := time.Parse("2006-01-02T15:04:05-07:00", "2019-01-18T23:21:54-08:00")
		require.NoError(t, err)

		logEvent1 := LogEvent{
			ID:        "1234",
			Timestamp: createMilisecondUnixTimestamp(currentTime),
			Message:   `{"level": "info", "message": "hello world 1"}`,
		}

		logEvent2 := LogEvent{
			ID:        "5678",
			Timestamp: createMilisecondUnixTimestamp(currentTime),
			Message:   `{"level": "info", "message": "hello world 2"}`,
		}

		data := CloudwatchLogsMessageData{
			MessageType:         "DATA_MESSAGE",
			Owner:               "524332086747",
			LogGroup:            "testGroup",
			LogStream:           "testStream",
			SubscriptionFilters: []string {"TestShipper"},
			LogEvents:           []LogEvent {logEvent1, logEvent2, },
		}

		expectedOutput :=
`{"index":{"_index":"cwl-2019.01.18","_type":"cwl","_id":"1234"}}
{"@id":"1234","@log_group":"testGroup","@log_stream":"testStream","@message":"{\"level\": \"info\", \"message\": \"hello world 1\"}","@owner":"524332086747","@timestamp":"2019-01-18T23:21:54-08:00","level":"info","message":"hello world 1"}

{"index":{"_index":"cwl-2019.01.18","_type":"cwl","_id":"5678"}}
{"@id":"5678","@log_group":"testGroup","@log_stream":"testStream","@message":"{\"level\": \"info\", \"message\": \"hello world 2\"}","@owner":"524332086747","@timestamp":"2019-01-18T23:21:54-08:00","level":"info","message":"hello world 2"}

`

		result, err := data.ToElasticsearchBulkBody()
		assert.NoError(t, err)
		assert.Equal(t, expectedOutput, string(result))
	})
}

func TestLogEvent_ToElasticsearchBulkBody(t *testing.T) {
	t.Run("properly encodes", func (t *testing.T) {
		event := LogEvent{
			ID:        "1234",
			Timestamp: createMilisecondUnixTimestamp(time.Now()),
			Message:   "{\"message\": \"this is a test log\"}",
		}

		payload := &CloudwatchLogsMessageData{
			MessageType:         "DATA_MESSAGE",
			Owner:               "524332086747",
			LogGroup:            "testGroup",
			LogStream:           "testStream",
			SubscriptionFilters: []string {"TestShipper"},
			LogEvents:           []LogEvent { event, },
		}

		// convert the message to json
		messageJson, err := json.Marshal(event.Message)
		require.NoError(t, err)

		// a lot of string parsing... sorry for who needs to read this...
		expectedMessage := fmt.Sprintf("{\"@id\":\"%s\",\"@log_group\":\"%s\",\"@log_stream\":\"%s\",\"@message\":%s,\"@owner\":\"%s\",\"@timestamp\":\"%s\",\"message\":\"this is a test log\"}",
			event.ID, payload.LogGroup, payload.LogStream, messageJson, payload.Owner, event.GetParsedTimestamp())
		expected := fmt.Sprintf("{\"index\":{\"_index\":\"%s\",\"_type\":\"cwl\",\"_id\":\"%s\"}}\n%s\n",
			GetEsIndex(event.Timestamp), event.ID, expectedMessage)

		result, err := event.ToElasticsearchBulkBody(payload)
		assert.NoError(t, err)
		assert.Equal(t, expected, string(result))
	})

	t.Run("handles non-json input by inserting it raw into @message", func (t *testing.T) {
		event := LogEvent{
			ID:        "1234",
			Timestamp: createMilisecondUnixTimestamp(time.Now()),
			Message:   "this is not a json message",
		}

		payload := &CloudwatchLogsMessageData{
			MessageType:         "DATA_MESSAGE",
			Owner:               "524332086747",
			LogGroup:            "testGroup",
			LogStream:           "testStream",
			SubscriptionFilters: []string {"TestShipper"},
			LogEvents:           []LogEvent { event, },
		}

		res, err := event.ToElasticsearchBulkBody(payload)
		assert.NoError(t, err)
		assert.NotEmpty(t, res)
		containsText := fmt.Sprintf("\"@message\":\"%s\"", event.Message)
		assert.True(t, strings.Contains(string(res), containsText),
			fmt.Sprintf("String did not contain substring.\nFull string: %s\nSub String: %s", string(res), containsText))
	})

	t.Run("handles invalid UTF-8 BOM Characters by inserting it raw into @message", func (t *testing.T) {
		event := LogEvent{
			ID:        "1234",
			Timestamp: createMilisecondUnixTimestamp(time.Now()),
			Message:   `ï{"message": "this is a test log"}`,
		}

		payload := &CloudwatchLogsMessageData{
			MessageType:         "DATA_MESSAGE",
			Owner:               "524332086747",
			LogGroup:            "testGroup",
			LogStream:           "testStream",
			SubscriptionFilters: []string {"TestShipper"},
			LogEvents:           []LogEvent { event, },
		}

		res, err := event.ToElasticsearchBulkBody(payload)
		assert.NoError(t, err)
		assert.NotEmpty(t, res)
		containsText := `"@message":"ï{\"message\": \"this is a test log\"}"`
		assert.True(t, strings.Contains(string(res), containsText),
			fmt.Sprintf("String did not contain substring.\nFull string: %s\nSub String: %s", string(res), containsText))
	})
}

func TestGetCwEsIndex(t *testing.T) {
	t.Run("returns the proper format", func (t *testing.T) {
		// create a Time object for Jan 2nd 2006
		currentTime, err := time.Parse("2006-01-02T15:04:05-07:00", "2019-01-18T23:21:54-08:00")
		require.NoError(t, err)

		result := GetEsIndex(createMilisecondUnixTimestamp(currentTime))
		assert.Equal(t, "cwl-2019.01.18", result)
	})
}

func TestLogEvent_GetParsedTimestamp(t *testing.T) {
	t.Run("properly parses", func (t *testing.T) {
		currentTimeStr := "2019-01-18T23:21:54-08:00"
		currentTime, err := time.Parse("2006-01-02T15:04:05-07:00", currentTimeStr)
		require.NoError(t, err)

		event := LogEvent{Timestamp: createMilisecondUnixTimestamp(currentTime), }

		assert.Equal(t, currentTimeStr, event.GetParsedTimestamp())
	})
}

// AWS Sends us Milliseconds Since Unix. Go doesn't have a UnixMili, so convert it
func createMilisecondUnixTimestamp(currentTime time.Time) int64 {
	return currentTime.UnixNano() / int64(time.Millisecond)
}
