package main

import (
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"os"
	"testing"
)

func TestNewConfig(t *testing.T) {
	t.Run("when elasticsearch endpoint is not provided returns an error", func (t *testing.T) {
		_, err := NewConfig()
		assert.EqualError(t, err, fmt.Sprintf("missing required environment variable: %s", EnvElasticsearchendpoint))
	})

	t.Run("when es endpoint provided, uses that", func (t *testing.T) {
		esEndpoint := "https://test.com"
		err := os.Setenv(EnvElasticsearchendpoint, esEndpoint)
		defer os.Unsetenv(EnvElasticsearchendpoint)
		require.NoError(t, err)

		config, err := NewConfig()
		assert.NoError(t, err)
		assert.Equal(t, esEndpoint, config.ElasticsearchEndpoint)
	})

	t.Run("when region not provided, returns default", func (t *testing.T) {
		err := os.Setenv(EnvElasticsearchendpoint, "http://test.com")
		defer os.Unsetenv(EnvElasticsearchendpoint)
		require.NoError(t, err)

		config, err := NewConfig()
		assert.NoError(t, err)
		assert.Equal(t, DefaultAwsRegion, config.Region)
	})

	t.Run("when aws region is set, uses that", func (t *testing.T) {
		err := os.Setenv(EnvElasticsearchendpoint, "http://test.com")
		defer os.Unsetenv(EnvElasticsearchendpoint)
		require.NoError(t, err)

		region := "us-no-exist"
		err = os.Setenv(EnvAwsRegion, region)
		defer os.Unsetenv(EnvAwsRegion)
		require.NoError(t, err)

		config, err := NewConfig()
		assert.NoError(t, err)
		assert.Equal(t, region, config.Region)
	})
}
