package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/elastic/go-elasticsearch/v7"
	"log"
)

const controlMessage = "CONTROL_MESSAGE" // Used by Cloudwatch Logs

func main() {
	lambda.Start(Handler)
}

// Handles an Event from AWS Cloudwatch Logs and Transfers it to Elasticsearch
func Handler(event Event) error {
	// Create Configuration
	config, err := NewConfig()
	if err != nil { return err }

	// Create ElasticSearch Client
	es, err := setupESClient(config.ElasticsearchEndpoint, config)
	if err != nil { return err }

	// Take the Base 64 Zip Encoded Data from CW Logs and decode it
	cwDecodedRawData, err := event.DecodeData()
	if err != nil { return err }

	// Convert JSON Raw Data to Strut so we can work with it
	cwData := &CloudwatchLogsMessageData{}
	err = json.Unmarshal(cwDecodedRawData, cwData)
	if err != nil { return err }
	log.Printf("Cloudwatch Logs Message Data: %+v", cwData)

	// Skip shipping if it's a control message
	if cwData.MessageType == controlMessage {
		log.Printf("[INFO] Received a control message. Skipping the ship to Elasticsearch")
		return nil
	}

	// Prepare to ship the data to ElasticSearch
	// If an error was found, raise it later on so other events in the same batch can be sent
	dataToSend, errorFound := cwData.ToElasticsearchBulkBody()
	if errorFound != nil {
		return err
	}
	log.Printf("The Data to send:\n%s", string(dataToSend))

	// Ship to ElasticSearch
	err = bulkPostToElasticsearch(es, dataToSend)
	if err != nil { return err }

	// Raise the delayed error found from building the bulk body
	if errorFound != nil {
		return errorFound
	}

	return nil
}

// Takes a body of pre-formatted data and sends it to the Elasticsearch Bulk Post API
// Formatting: https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-bulk.html
func bulkPostToElasticsearch(esClient *elasticsearch.Client, body []byte) error {
	strReader := bytes.NewReader(body)

	resp, err := esClient.Bulk(strReader)
	if err != nil {
		return fmt.Errorf("problem posting data to elasticsearch: %v", err)
	}

	log.Printf("The response from ES: %+v\n", resp)

	// If the status is not a 2xx, return an error
	if resp.StatusCode < 200 || resp.StatusCode >= 300 {
		return fmt.Errorf("elasticsearch returned unknown status code: %d %s", resp.StatusCode, resp.Status())
	}

	return nil
}
