variable "elasticsearch_endpoint_url" {} // The Elasticsearch Service Endpoint URL to ship logs to
variable "elasticsearch_arn" {} // The Elasticsearch Service ARN to ship logs to
variable "timeout" { default = 2  } // the max amount of time to run in seconds
variable "reserved_concurrent_executions" { default = 20 } // The amount of reserved concurrent executions for this lambda function

# Create a Random ID - Avoid Conflicts incase an account has multiple of these lambdas created
resource "random_id" "id" {
  byte_length = 6
}

# Create a Log Group for the Lambda to Use
resource "aws_cloudwatch_log_group" "cw_es_shipper" {
  name              = "/aws/lambda/cloudwatch_elasticsearch_shipper_${random_id.id.id}"
  retention_in_days = 90
}

# Lambda Function for ES Shipper
resource "aws_lambda_function" "cw_es_shipper" {
  // Would prefer to use S3 Bucket/Key instead of storing the file in git
  // However, S3 bucket permissions would need to be provided to anyone who runs this script
  // The only way I can think of is to only run Terraform in CI, and give that permission
  // However, that still complicates testings and would be a large project, so using Filename for now
  filename         = "${path.module}/cw_es_shipper.zip"
  description      = "Ships Cloudwatch Logs to an Elastic Search Cluster"
  function_name    = "cloudwatch_elasticsearch_shipper_${random_id.id.id}"
  role             = "${aws_iam_role.cw_es_shipper.arn}"
  handler          = "cw_es_shipper"
  source_code_hash = "${filebase64sha256("${path.module}/cw_es_shipper.zip")}"
  timeout          = "${var.timeout}"
  reserved_concurrent_executions = "${var.reserved_concurrent_executions}"

  runtime = "go1.x"

  environment {
    variables = {
      ES_ENDPOINT = "${var.elasticsearch_endpoint_url}"
    }
  }

  // use tags to help with cost explorer
  tags {
    Service = "cwl-es-shipper"
  }
}

output "aws_lambda_function_name" {
  value = "${aws_lambda_function.cw_es_shipper.function_name}"
}
