package main

import (
	"errors"
	"fmt"
	"os"
	"strconv"
	"strings"
	"time"
)

const (
	esEndpointENV    = "ES_ENDPOINT"
	retentionDaysENV = "RETENTION_DAYS"
	localModeENV     = "LOCAL"
	dryRunENV        = "DRY_RUN"
	regionENV        = "AWS_DEFAULT_REGION"
	defaultRegion    = "us-west-2"
	defaultRetention = (time.Hour * 24) * 30
)

type Config struct {
	Endpoint  string
	Retention time.Duration
	LocalMode bool
	DryRun    bool
	Region    string
}

// Creates a configuration based on provided environment variables
func NewConfig() (*Config, error) {
	c := &Config{}

	// Set ElasticSearch Endpoint
	esEndpoint, err := getEsEndpoint()
	if err != nil { return c, err }
	c.Endpoint = esEndpoint

	// Set Retention Days
	retentionDays, err := getRetentionDays()
	if err != nil { return c, err }
	c.Retention = retentionDays

	// Set LocalMode
	c.LocalMode = isLocalMode()

	// Set DryRun
	c.DryRun = isDryRun()

	// Set Region
	c.Region = getRegion()

	return c, nil
}

// Returns the user specified ElasticSearch Endpoint
// Returns an error if no environment variable was provided by the user
func getEsEndpoint() (string, error) {
	endpoint := os.Getenv(esEndpointENV)
	if endpoint == "" {
		return endpoint, fmt.Errorf("%s is a required environment variable", esEndpointENV)
	}
	return endpoint, nil
}

// Returns the number of retention days, or the default if not provided
// Returns an error if there was a problem parsing, or the user supplied an unsupported input
func getRetentionDays() (time.Duration, error) {
	retentionDaysStr := os.Getenv(retentionDaysENV)
	if retentionDaysStr == "" {
		return defaultRetention, nil
	}

	retentionDays, err := strconv.Atoi(retentionDaysStr)
	if err != nil {
		return defaultRetention, err
	}

	// We don't currently support factoring in anything less than a day
	if retentionDays <= 0 {
		return defaultRetention, errors.New("a retention 0 or less was provided")
	}

	return (time.Hour * 24) * time.Duration(retentionDays), nil
}

// Returns if the user requested to run in "local" (non-lambda) mode
func isLocalMode() bool {
	return strings.ToLower(os.Getenv(localModeENV)) == "true"
}

// Returns if the user wants to run this without actually causing delete actions
func isDryRun() bool {
	return strings.ToLower(os.Getenv(dryRunENV)) == "true"
}

func getRegion() string {
	region := os.Getenv(regionENV)
	if region == "" {
		return defaultRegion
	}
	return region
}
