package main

import (
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"os"
	"testing"
	"time"
)

func TestNewConfig(t *testing.T) {
	t.Run("returns config as set", func (t *testing.T) {
		esEndpoint := "https://es.endpoint.aws.test"
		os.Setenv(esEndpointENV, esEndpoint)
		defer os.Unsetenv(esEndpointENV)

		retentionDays := "30"
		retentionDaysDuration := (time.Hour * 24) * 30
		os.Setenv(retentionDaysENV, retentionDays)
		defer os.Unsetenv(retentionDaysENV)

		localMode := "false"
		os.Setenv(localModeENV, localMode)
		defer os.Unsetenv(localModeENV)

		dryRun := "false"
		os.Setenv(dryRunENV, dryRun)
		defer os.Unsetenv(dryRunENV)

		region := "fooregion"
		os.Setenv(regionENV, region)
		defer os.Unsetenv(regionENV)

		config, err := NewConfig()
		assert.NoError(t, err)
		assert.NotNil(t, config)

		assert.Equal(t, esEndpoint, config.Endpoint)
		assert.Equal(t, retentionDaysDuration, config.Retention)
		assert.Equal(t, region, config.Region)

		// Convert the strings to bools
		assert.Equal(t, localMode == "true", config.LocalMode)
		assert.Equal(t, dryRun == "true", config.DryRun)
	})

	t.Run("uses defaults as necessary", func (t *testing.T) {
		os.Setenv(esEndpointENV, "http://test")
		defer os.Unsetenv(esEndpointENV)

		config, err := NewConfig()
		assert.NoError(t, err)
		assert.NotNil(t, config)
		assert.Equal(t, config.Retention, defaultRetention)
		assert.Equal(t, config.LocalMode, false)
		assert.Equal(t, config.DryRun, false)
		assert.Equal(t, config.Region, defaultRegion)
	})

	t.Run("required is required", func (t *testing.T) {
		defer os.Unsetenv(esEndpointENV)

		_, err := NewConfig()
		assert.EqualError(t, err, fmt.Sprintf("%s is a required environment variable", esEndpointENV))

		esEndpoint := "https://es.endpoint.aws.test"
		os.Setenv(esEndpointENV, esEndpoint)
		config, err := NewConfig()
		assert.NoError(t, err)
		assert.Equal(t, esEndpoint, config.Endpoint)
	})
}

func TestGetRetentionDays(t *testing.T) {
	defer os.Unsetenv(retentionDaysENV)

	var intDurationTests = []struct {
		in     string
		out    time.Duration
		outErr bool // if there was an error returned
	}{
		{"1", time.Hour * 24, false},
		{"30", (time.Hour * 24) * 30, false},
		{"", defaultRetention, false},
		{"0", defaultRetention, true},
		{"-1", defaultRetention, true},
		{"notanint", defaultRetention,true},
	}

	for _, tt := range intDurationTests {
		t.Run(tt.in, func (t *testing.T) {
			err := os.Setenv(retentionDaysENV, tt.in)
			require.NoError(t, err)

			res, err := getRetentionDays()
			assert.Equal(t, tt.out, res)
			assert.Equal(t, tt.outErr, err != nil)
		})
	}
}

func TestGetRegion(t *testing.T) {
	defer os.Unsetenv(regionENV)

	var regionTest = []struct {
		in string
		out string
	}{
		{"", defaultRegion},
		{"fooregion", "fooregion"},
	}

	for _, tt := range regionTest {
		t.Run(tt.in, func (t *testing.T) {
			os.Setenv(regionENV, tt.in)
			assert.Equal(t, tt.out, getRegion())
		})
	}
}
