package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/aws/signer/v4"
	"io/ioutil"
	"log"
	"net/http"
	"sort"
	"strings"
	"time"
)

type ElasticSearch struct {
	Endpoint string
	Region   string
	Signer   *v4.Signer
}

const httpClientTimeout = time.Second * 30

// Entry point
func main() {
	if isLocalMode() {
		err := HandleRequest()
		if err != nil {
			log.Fatalf("While running, received error: %v", err)
		}
	} else {
		lambda.Start(HandleRequest)
	}
}

// Handler for the Lambda request
func HandleRequest() error {
	config, err := NewConfig()
	if err != nil { return err }

	// Subtract the retention days from the current time
	// Using the .Add() command as a subtract isn't available. Subtracting by adding a negative number of days
	deleteBefore := time.Now().Add(- config.Retention)
	return QueryAndDeleteOldIndexes(config, deleteBefore)
}

// Function that will query and delete old indexes
// c is the configuration object for the program
// deleteBefore is the time at which indexes should be deleted on or before that date
func QueryAndDeleteOldIndexes(c *Config, deleteBefore time.Time) error {
	es := ElasticSearch{Endpoint: c.Endpoint, Region: c.Region}

	// Set up a signer for Amazon Signed Requests
	sess := session.Must(session.NewSession())
	es.Signer = v4.NewSigner(sess.Config.Credentials)

	log.Printf("Deleting Indexes Before: %s", deleteBefore)

	// Fetch all ES indices so we can analyze them
	beforeIndices, err := es.GetIndices()
	if err != nil {
		return err
	}
	log.Printf("All indices returned from ElasticSearch: %s", beforeIndices)

	// Loop through them, and delete them if they should be deleted
	errorDetected := false // store a flag for an error so we can raise an error upon completion
	for _, esIndex := range beforeIndices {
		if ShouldDeleteIndex(esIndex, deleteBefore) {
			err = es.DeleteIndex(esIndex)
			if err != nil {
				// Silently warn and continue
				errorDetected = true
				log.Printf("WARN: Encountered error deleting index [%s]: %v", esIndex, err)
			}
		}
	}
	if errorDetected {
		return errors.New("there was an error deleting an index - check the logs")
	}

	// Fetch indices again and display the before/after effect of running the delete
	log.Println("Completed deleting indices past retention")
	afterIndices, _ := es.GetIndices()

	// Sort just for clarity
	sort.Strings(beforeIndices)
	sort.Strings(afterIndices)
	log.Printf("Before and After:\nB: %s\nA: %s", beforeIndices, afterIndices)
	return nil
}

// Returns a bool on if an index should be deleted
func ShouldDeleteIndex(index string, retention time.Time) bool {
	const prefix = "cwl-"

	if !strings.HasPrefix(index, prefix) {
		log.Printf("Skipping %s. The prefix %s was not contained within the index", index, prefix)
		return false
	}

	const shortForm = "cwl-2006.01.02"
	timeObj, err := time.Parse(shortForm, index)
	if err != nil {
		log.Fatalf("Encountered error: %v", err)
		return false
	}

	return timeObj.Before(retention)
}

// Returns an array of all ElasticSearch Indices
func (e *ElasticSearch) GetIndices() ([]string, error) {
	var results []string
	endpoint := fmt.Sprintf("%s/_aliases", e.Endpoint)

	c := http.Client{
		Timeout: httpClientTimeout,
	}
	req, err := http.NewRequest(http.MethodGet, endpoint, nil)
	if err != nil {
		return results, err
	}

	// Sign the request for AWS
	err = e.signRequest(req)
	if err != nil {
		return results, err
	}

	resp, err := c.Do(req)
	if err != nil {
		return results, err
	}

	if resp.StatusCode != http.StatusOK {
		return results, fmt.Errorf("received status code: %d", resp.StatusCode)
	}

	// Read the body, unmarshal the json
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return results, err
	}

	var indices map[string]interface{}
	err = json.Unmarshal(body, &indices)
	if err != nil {
		return results, err
	}

	// Append the results into an array
	for key, _ := range indices {
		results = append(results, key)
	}

	return results, nil
}

// Deletes an index from ElasticSearch
// Provide the index name to delete
func (e *ElasticSearch) DeleteIndex(index string) error {
	log.Printf("Deleting index: %s", index)
	if isDryRun() { return nil }

	client := &http.Client{
		Timeout: httpClientTimeout,
	}

	req, err := http.NewRequest(http.MethodDelete, fmt.Sprintf("%s/%s", e.Endpoint, index), nil)
	if err != nil { return err }

	err = e.signRequest(req)
	if err != nil {
		return err
	}

	resp, err := client.Do(req)
	if err != nil { return err }

	// If the status was not successful, return an error with the body of the response
	if resp.StatusCode != http.StatusOK {
		errorBase := fmt.Sprintf("encountered error deleting index. Returned status code %d", resp.StatusCode)

		defer resp.Body.Close()
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return fmt.Errorf("%s. Error reading body: %v", errorBase, err)
		}

		return fmt.Errorf("%s. Body: %s", errorBase, string(body))
	}

	return nil
}

// Signs an http request for AWS to ensure permissions are correct
func (e *ElasticSearch) signRequest(req *http.Request) error {
	// Sign the request for AWS
	if e.Signer != nil {
		_, err := e.Signer.Sign(req, nil, "es", e.Region, time.Now())
		return err
	}
	return nil
}
