package main

import (
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"net/http"
	"net/http/httptest"
	"sort"
	"strings"
	"testing"
	"time"
)

func TestShouldDeleteIndex(t *testing.T) {
	timeLayoutShortForm := "2006-Jan-02"
	retention, err := time.Parse(timeLayoutShortForm, "2020-Jan-15")
	require.NoError(t, err)

	var retentionTests = []struct {
		in string
		out bool
	}{
		{".kibana-1", false}, // non cwl prefix should not be deleted
		{"cwl-2020.01.14", true}, // day before should be deleted
		{"cwl-2020.01.15", false}, // day on should not be deleted
		{"cwl-2020.01.16", false}, // day after should not be deleted
	}

	for _, tt := range retentionTests {
		t.Run(tt.in, func (t *testing.T) {
			assert.Equal(t, tt.out, ShouldDeleteIndex(tt.in, retention))
		})
	}
}

func TestElasticSearch_GetIndices(t *testing.T) {
	t.Run("when successful", func (t *testing.T) {
		indexes := []string{
			".kibana_1",
			"cwl-2020.01.24",
			"cwl-2020.01.25",
		}

		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprint(w, mockESResponse(indexes))
		}))
		defer server.Close()

		es := ElasticSearch{Endpoint: server.URL}

		res, err := es.GetIndices()
		assert.NoError(t, err)
		assert.NotEmpty(t, res)

		sort.Strings(res)
		assert.Equal(t, indexes, res)

	})

	t.Run("returns an error when an error", func (t *testing.T) {
		errorMsg := "there was an error"
		errorCode := http.StatusInternalServerError

		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			http.Error(w, errorMsg, errorCode)
		}))
		defer server.Close()

		es := ElasticSearch{Endpoint: server.URL}
		res, err := es.GetIndices()
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("received status code: %d", errorCode))
		assert.Empty(t, res)
	})
}

func TestElasticSearch_DeleteIndex(t *testing.T) {
	t.Run("when successful", func (t *testing.T) {
		requests := 0
		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			requests += 1
			w.WriteHeader(http.StatusOK)
		}))
		defer server.Close()

		es := ElasticSearch{Endpoint: server.URL}
		err := es.DeleteIndex("cwl-2020.01.24")
		assert.NoError(t, err)
		assert.Equal(t, 1, requests, "a request should have reached the server")
	})

	t.Run("when the server returns an error the function returns an error", func (t *testing.T) {
		errorMsg := "error deleting"
		errorCode := http.StatusInternalServerError
		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			http.Error(w, errorMsg, errorCode)
		}))
		defer server.Close()

		es := ElasticSearch{Endpoint: server.URL}
		err := es.DeleteIndex("cwl-2020.01.24")
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("encountered error deleting index. Returned status code %d. Body: %s\n", errorCode, errorMsg))
	})
}

func TestDeleteOldIndexes(t *testing.T) {
	retention, err := time.Parse("2006-Jan-02", "2020-Jan-25")
	require.NoError(t, err)

	// Create variable of indexes - copies of each other for comparison, but indexesModified will be modified by server
	indexesOriginal := []string{".kibana_1", "cwl-2020.01.25", "cwl-2020.01.24", "cwl-2020.01.23"}
	indexesModified := []string{".kibana_1", "cwl-2020.01.25", "cwl-2020.01.24", "cwl-2020.01.23"}

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.Method == http.MethodGet { // return the indexes
			fmt.Fprint(w, mockESResponse(indexesModified))
			return
		} else if r.Method == http.MethodDelete { // delete the index
			indexToDelete := strings.TrimPrefix(r.URL.Path, "/")

			// Look through all indexes
			for key, value := range indexesModified {
				if value == indexToDelete { // delete the match
					indexesModified = append(indexesModified[:key], indexesModified[key+1:]...)
					return
				}
			}
		} else {
			http.Error(w, "not implemented", http.StatusNotImplemented)
		}
	}))
	defer server.Close()

	assert.Equal(t, indexesOriginal, indexesModified)
	assert.Contains(t, indexesModified, "cwl-2020.01.24")
	assert.Contains(t, indexesModified, "cwl-2020.01.23")

	// Run the Command
	config := &Config{Endpoint: server.URL, Region: defaultRegion}
	err = QueryAndDeleteOldIndexes(config, retention)
	assert.NoError(t, err)

	// Ensure the proper items were deleted
	assert.NotEqual(t, indexesOriginal, indexesModified)
	assert.NotContains(t, indexesModified, "cwl-2020.01.24")
	assert.NotContains(t, indexesModified, "cwl-2020.01.23")

}

// Creastes a json response of what ElasticSearch returns for indexes
// Supply an array of the indexes to return
// Example response:
// {".kibana_1":{"aliases":{".kibana":{}}},"cwl-2020.01.25":{"aliases":{".kibana":{}}},"cwl-2020.01.24":{"aliases":{".kibana":{}}}}
func mockESResponse(indexes []string) string {
	response := "{"
	for key, value := range indexes {
		response = response + `"` + value + `":{"aliases":{".kibana":{}}}`

		if key != len(indexes) - 1 {
			response = response + ","
		}
	}
	response = response + "}"
	return response
}
