package check

import (
	"go/token"
	"path/filepath"
	"regexp"
	"strings"

	"code.justin.tv/tshadwell/nice"
	"code.justin.tv/tshadwell/nice/naiive"
)

// Naiive is the check for the GAS Go security-oriented static code analyser
type Naiive struct {
	Dir       string
	Filenames []string
}

// Name returns the name of the display name of the command
func (Naiive) Name() string { return "naiive" }

// Weight returns the weight this check has in the overall average
func (Naiive) Weight() float64 { return 0 }

var reWorthwhilePath = regexp.MustCompile("(?:.*/src/)(.*)")

// naiive uses full filepaths. This strips them down to import paths.
func normalisePath(path string) (newPath string) {
	return reWorthwhilePath.FindStringSubmatch(path)[1]
}

// Percentage returns the percentage of .go files that pass naiive
func (g Naiive) Percentage() (score float64, summaries []FileSummary, err error) {
	var p nice.Program

	if err = p.Import("./" + filepath.Join(g.Dir, "...")); err != nil {
		return
	}

	cfg, err := p.Config()
	if err != nil {
		return
	}

	cfg.AllowErrors = true

	results, err := naiive.DefaultTests.
		ConfidenceRange(nice.MediumConfidence, -1).
		SeverityRange(nice.MediumSeverity, -1).
		NiceTest(&p)

	if err != nil {
		return
	}

	var filesMap = make(map[string][]Error)

	for _, result := range results {
		filename := normalisePath(result.Position[0].Filename)
		filesMap[filename] = append(filesMap[filename], Error{
			LineNumber:  result.Position[0].Line,
			ErrorString: strings.Replace(result.TestDescription(), "\n", "\n\n", -1),
		})
	}

	for fileName, errors := range filesMap {
		summaries = append(summaries, FileSummary{
			Filename: fileName,
			FileURL:  fileURL(g.Dir, "/"+fileName),
			Errors:   errors,
		})
	}

	var filecount uint
	// count files
	cfg.Fset.Iterate(func(*token.File) bool { filecount++; return true })

	score = 1 - float64(len(summaries))/float64(filecount)

	return
}

// Description returns the description of gas
func (Naiive) Description() string {
	return `<code>naiive</code> examines Go source code and reports potential security issues with varying levels of confidence.`
}
