package check

import (
	"reflect"
	"testing"
)

func TestGoFiles(t *testing.T) {
	files, skipped, err := GoFiles("testfiles/")
	if err != nil {
		t.Fatal(err)
	}
	want := []string{"testfiles/a.go", "testfiles/b.go", "testfiles/c.go"}
	if !reflect.DeepEqual(files, want) {
		t.Errorf("GoFiles(%q) = %v, want %v", "testfiles/", files, want)
	}

	wantSkipped := []string{"testfiles/a.pb.go", "testfiles/a.pb.gw.go"}
	if !reflect.DeepEqual(skipped, wantSkipped) {
		t.Errorf("GoFiles(%q) skipped = %v, want %v", "testfiles/", skipped, wantSkipped)
	}
}

var goToolTests = []struct {
	name      string
	dir       string
	filenames []string
	tool      []string
	percent   float64
	failed    []FileSummary
	wantErr   bool
}{
	{"go vet", "testfiles/", []string{"testfiles/a.go", "testfiles/b.go", "testfiles/c.go"}, []string{"go", "tool", "vet"}, 1, []FileSummary{}, false},
}

func TestGoTool(t *testing.T) {
	for _, tt := range goToolTests {
		f, fs, err := GoTool(tt.dir, tt.filenames, tt.tool)
		if err != nil && !tt.wantErr {
			t.Fatal(err)
		}
		if f != tt.percent {
			t.Errorf("[%s] GoTool percent = %f, want %f", tt.name, f, tt.percent)
		}
		if !reflect.DeepEqual(fs, tt.failed) {
			t.Errorf("[%s] GoTool failed = %v, want %v", tt.name, fs, tt.failed)
		}
	}
}

func TestAddParamsFromJSON(t *testing.T) {
	var params []string
	newParams, err := addParamsFromConfigFile(params, "testfiles")
	expected := []string{"--skip=testfiles/skipped/directory", "--exclude='exclude*'"}
	if !reflect.DeepEqual(newParams, expected) || err != nil {
		t.Error(newParams)
		t.Fatalf("params do not match")
	}
}

func TestAddParamsFromConfig(t *testing.T) {
	var startParams []string

	newParams, err := addParamsFromJSON(startParams, []byte{}, "testfiles")
	if len(newParams) != len(startParams) || err != nil {
		t.Fatalf("added params with no config")
	}

	config := `
{
  "Deadline": "3m",
  "Enable": [
    "errcheck",
    "goimports",
    "gofmt",
    "ineffassign",
    "misspell",
    "megacheck",
    "vet"
  ],
  "Linters": {
    "megacheck": {
      "Command": "megacheck -simple.exit-non-zero=true"
    }
  },
  "Skip": [
    "internal/clients/mocks",
    "internal/clients/wrappers"
  ],
  "Vendor": true
}
`
	newParams, err = addParamsFromJSON(startParams, []byte(config), "testfiles")
	expected := []string{"--skip=testfiles/internal/clients/mocks", "--skip=testfiles/internal/clients/wrappers"}
	if !reflect.DeepEqual(newParams, expected) || err != nil {
		t.Errorf("%v", newParams)
		t.Fatalf("config params skip were not successfully added")
	}

	config = `
{
    "Concurrency": 3,
    "Deadline": "5m",
    "EnableAll": true,
    "Disable": [
        "gocyclo",
        "duplo",
        "gas",
        "gotype",
        "interfacer"
    ],
    "Exclude": [
        "mock_.*\\.go",
        ".*\\.twirp\\.go"
    ],
    "Vendor": true
}`

	newParams, err = addParamsFromJSON(startParams, []byte(config), "testfiles")
	expected = []string{"--exclude='mock_.*\\.go'", "--exclude='.*\\.twirp\\.go'"}
	if !reflect.DeepEqual(newParams, expected) || err != nil {
		t.Fatalf("config params exclude were not successfully added")
	}
}
