package main

import (
	"flag"
	"fmt"
	"log"
	"net/http"
	"os"
	"regexp"
	"time"

	"golang.org/x/net/context"
	"golang.org/x/oauth2"

	"code.justin.tv/qe/goreportcard/handlers"

	"code.justin.tv/qe/goreportcard/db"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/middleware"
	noncePkg "code.justin.tv/systems/guardian/nonce"
	"github.com/boltdb/bolt"
	gorilla "github.com/gorilla/context"
	"github.com/gorilla/sessions"
	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promhttp"
	"github.com/sirupsen/logrus"
)

type contextKey string

var (
	addr               = flag.String("http", ":8000", "HTTP listen address")
	dev                = flag.Bool("dev", false, "dev mode")
	cookieStore        = sessions.NewCookieStore([]byte("1dkjshakj12h3kjhasdkjh12"))
	accessCookieName   = "access"
	refreshCookieName  = "refresh"
	guardianContextKey = contextKey("guardianUser")
)

const authCookieName = "auth-session"
const authBucketName = "auth"

func makeHandler(name string, dev bool, fn func(http.ResponseWriter, *http.Request, string, bool)) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		validPath := regexp.MustCompile(fmt.Sprintf(`^/%s/([a-zA-Z0-9\-_\/\.]+)$`, name))

		m := validPath.FindStringSubmatch(r.URL.Path)

		if m == nil {
			http.NotFound(w, r)
			return
		}
		if len(m) < 1 || m[1] == "" {
			http.Error(w, "Please enter a repository", http.StatusBadRequest)
			return
		}

		repo := m[1]

		// for backwards-compatibility, we must support URLs formatted as
		//   /report/[org]/[repo]
		// and they will be assumed to be github.com URLs. This is because
		// at first Go Report Card only supported github.com URLs, and
		// took only the org name and repo name as parameters. This is no longer the
		// case, but we do not want external links to break.
		oldFormat := regexp.MustCompile(fmt.Sprintf(`^/%s/([a-zA-Z0-9\-_]+)/([a-zA-Z0-9\-_]+)$`, name))
		m2 := oldFormat.FindStringSubmatch(r.URL.Path)
		if m2 != nil {
			// old format is being used
			repo = "github.com/" + repo
			log.Printf("Assuming intended repo is %q, redirecting", repo)
			http.Redirect(w, r, fmt.Sprintf("/%s/%s", name, repo), http.StatusMovedPermanently)
			return
		}

		fn(w, r, repo, dev)
	}
}

// initDB opens the bolt database file (or creates it if it does not exist), and creates
// a bucket for saving the repos, also only if it does not exist.
func initDB() (e error) {
	dbConnection, err := db.Connect()
	if err != nil {
		return err
	}
	defer func() {
		if err := dbConnection.Close(); err != nil {
			e = err
		}
	}()

	err = dbConnection.Update(func(tx *bolt.Tx) error {
		_, err = tx.CreateBucketIfNotExists([]byte(handlers.RepoBucket))
		if err != nil {
			return err
		}
		_, err = tx.CreateBucketIfNotExists([]byte(handlers.MetaBucket))
		return err
	})
	return err
}

// metrics provides functionality for monitoring the application status
type metrics struct {
	responseTimes *prometheus.SummaryVec
}

// setupMetrics creates custom Prometheus metrics for monitoring
// application statistics.
func setupMetrics() *metrics {
	m := &metrics{}
	m.responseTimes = prometheus.NewSummaryVec(
		prometheus.SummaryOpts{
			Name: "response_time_ms",
			Help: "Time to serve requests",
		},
		[]string{"endpoint"},
	)

	prometheus.MustRegister(m.responseTimes)
	return m
}

// recordDuration records the length of a request from start to now
func (m metrics) recordDuration(start time.Time, name string) {
	elapsed := time.Since(start)
	m.responseTimes.WithLabelValues(name).Observe(float64(elapsed.Nanoseconds()))
	log.Printf("Served %s in %s", name, elapsed)
}

// instrument adds metric instrumentation to the handler passed in as argument
func (m metrics) instrument(path string, h http.HandlerFunc) (string, http.HandlerFunc) {
	return path, func(w http.ResponseWriter, r *http.Request) {
		defer m.recordDuration(time.Now(), path)
		h.ServeHTTP(w, r)
	}
}

type middleware func(http.HandlerFunc) http.HandlerFunc

// setGuardianUser sets a value for this package in the
// request values.
func setGuardianUser(r *http.Request, val *guardian.User) *http.Request {
	ctx := context.WithValue(r.Context(), guardianContextKey, val)
	return r.WithContext(ctx)
}

func setupAsiimov() (*asiimov.Asiimov, *oauth2.Config) {

	clientID := os.Getenv("OAUTH_CLIENT")
	clientSecret := os.Getenv("OAUTH_KEY")
	oauth2Config := &oauth2.Config{
		ClientID:     clientID,
		ClientSecret: clientSecret,
		Scopes:       []string{},
		Endpoint: oauth2.Endpoint{
			AuthURL:  asiimov.AuthURL,
			TokenURL: asiimov.TokenURL,
		},
	}

	logger := logrus.New()
	asi := asiimov.New(logger, oauth2Config)
	asi.CheckTokenURL = asiimov.CheckTokenURL
	asi.VerifyAccess = asi.DefaultVerifyAccess
	asi.NonceStore = noncePkg.NewMap(30 * time.Minute)
	return asi, oauth2Config
}

// MakeOAuth2RedirectHandler creates an http.Handler for the oauth2 redirect and stores the url for later redirect after auth.
func MakeOAuth2RedirectHandler(asi *asiimov.Asiimov, oauth *oauth2.Config) (handler http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		nonce, err := asi.NonceStore.Generate()
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			log.Println("Error in Oauth handler")
			log.Println(err.Error())
			return
		}
		// Store nonce with request url
		connection, err := db.Connect()
		if err != nil {
			http.Error(w, "Could not connect to session db", http.StatusInternalServerError)
			log.Println(err.Error())
			return
		}
		defer func() {
			if err := connection.Close(); err != nil {
				http.Error(w, "Could not close session db", http.StatusInternalServerError)
				log.Println(err.Error())
				return
			}
		}()

		err = connection.Update(func(tx *bolt.Tx) error {
			b, err := tx.CreateBucketIfNotExists([]byte(authBucketName))
			if err != nil {
				return fmt.Errorf("create bucket: %s", err)
			}
			err = b.Put([]byte(nonce), []byte(r.URL.Path))
			return err
		})
		if err != nil {
			http.Error(w, "Could not connect to session db", http.StatusInternalServerError)
			log.Println(err.Error())
			return
		}

		http.Redirect(w, r, oauth.AuthCodeURL(nonce, oauth2.AccessTypeOnline), http.StatusFound)
	})
}

// oAuth2Middleware returns middleware that will inject identity into the context
// based on an oauth2 token
func oAuth2Middleware(asi *asiimov.Asiimov, oauth *oauth2.Config) (fn middleware, err error) {
	if asi == nil {
		log.Fatal("Tried using oauth middleware without set asiimov")
	}
	fn = func(inner http.HandlerFunc) http.HandlerFunc {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			switch r.Method {
			case "OPTIONS":
				return
			}
			session, err := cookieStore.Get(r, authCookieName)
			if err != nil {
				log.Printf("Could not get session from request %v", err)
			}
			val := session.Values[accessCookieName]
			if _, ok := val.(string); ok {
				inner.ServeHTTP(w, r)
				return
			}

			user, err := asi.HandleGuardianRequest(r)
			if err != nil || user == nil {
				log.Printf("No Oauth Token attached. redirecting	")
				handler := MakeOAuth2RedirectHandler(asi, oauth)
				handler.ServeHTTP(w, r)
			}
			inner.ServeHTTP(w, setGuardianUser(r, user))
		})
	}
	return
}

func oauthErrHandler(err asiimov.Error, w http.ResponseWriter, _ *http.Request) {
	log.Println(err.Error())
	http.Error(w, err.Error(), http.StatusInternalServerError)
}

func makeTokenHandler(asi *asiimov.Asiimov) asiimov.TokenHandler {
	return func(token *oauth2.Token, w http.ResponseWriter, r *http.Request) {
		if token == nil {
			http.Error(w, "Received nil token from Guardian. Could not Auth.", http.StatusInternalServerError)
			return
		}
		session, err := cookieStore.Get(r, authCookieName)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		fullToken, err := asi.CheckToken(token)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		session.Values["login-time"] = time.Now().Unix()
		session.Values["login-uid"] = fullToken.User.UID
		session.Values["login-email"] = fullToken.User.Email
		session.Values[accessCookieName] = token.AccessToken
		session.Values[refreshCookieName] = token.RefreshToken
		err = session.Save(r, w)
		if err != nil {
			log.Printf("could not connect to database to get redirect")
			log.Println(err.Error())
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		connection, err := db.Connect()
		if err != nil {
			log.Printf("could not connect to database to get redirect")
			log.Println(err.Error())
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		defer func() {
			if err := connection.Close(); err != nil {
				log.Printf("could not close redirect db")
				log.Println(err.Error())
				http.Error(w, err.Error(), http.StatusInternalServerError)
			}
		}()

		redirectURL := "/"
		state := []byte(r.FormValue("state"))
		err = connection.Update(func(tx *bolt.Tx) error {
			b, err := tx.CreateBucketIfNotExists([]byte(authBucketName))
			if err != nil {
				return fmt.Errorf("create bucket: %s", err)
			}
			redirectURL = string(b.Get(state))
			err = b.Delete(state)
			return err
		})
		if err != nil {
			log.Printf("could not connect to database to get redirect")
			log.Println(err.Error())
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		http.Redirect(w, r, redirectURL, http.StatusFound)
	}
}

func logoutHandler(w http.ResponseWriter, r *http.Request) {
	session, err := cookieStore.Get(r, authCookieName)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	delete(session.Values, accessCookieName)
	delete(session.Values, refreshCookieName)
	err = session.Save(r, w)
	if err != nil {
		log.Println("unable to clear cookie because of")
		log.Println(err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}
	_, err = w.Write([]byte("Logged Out"))
	if err != nil {
		log.Println("unable to write logged out to response")
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}

}

func main() {
	flag.Parse()
	if err := os.MkdirAll("_repos/src/github.com", 0755); err != nil && !os.IsExist(err) {
		log.Fatal("ERROR: could not create repos dir: ", err)
	}

	// initialize database
	if err := initDB(); err != nil {
		log.Fatal("ERROR: could not open bolt db: ", err)
	}

	asi, oauthConfig := setupAsiimov()
	m := setupMetrics()
	oauthMiddleware, err := oAuth2Middleware(asi, oauthConfig)
	if err != nil {
		log.Fatal("Could not initialize middleware")
	}
	tokenHandler := makeTokenHandler(asi)

	http.HandleFunc(m.instrument("/assets/", handlers.AssetsHandler))
	http.HandleFunc(m.instrument("/favicon.ico", handlers.FaviconHandler))
	http.HandleFunc(m.instrument("/checks", handlers.CheckHandler))
	http.HandleFunc(m.instrument("/report/", oauthMiddleware(makeHandler("report", *dev, handlers.ReportHandler))))
	http.HandleFunc(m.instrument("/badge/", makeHandler("badge", *dev, handlers.BadgeHandler)))
	http.HandleFunc(m.instrument("/score/", makeHandler("score", *dev, handlers.ScoreHandler)))
	http.HandleFunc(m.instrument("/high_scores/", oauthMiddleware(handlers.HighScoresHandler)))
	http.HandleFunc(m.instrument("/about/", handlers.AboutHandler))
	http.HandleFunc(m.instrument("/", handlers.HomeHandler))
	http.HandleFunc(m.instrument("/oauth/callback", asi.OAuth2CallbackHandler(oauthErrHandler, tokenHandler).ServeHTTP))
	http.HandleFunc(m.instrument("/logout", logoutHandler))

	http.Handle("/metrics", promhttp.Handler())

	log.Printf("Running on %s ...", *addr)
	// from http://www.gorillatoolkit.org/pkg/sessions. fixes memory leaks
	log.Fatal(http.ListenAndServe(*addr, gorilla.ClearHandler(http.DefaultServeMux)))
}
