package main

import (
	"fmt"
	"io/ioutil"
	"log"

	"bufio"
	"bytes"
	"encoding/csv"
	"encoding/json"
	"flag"
	"net/http"
	"os"
	"strings"
	"time"
)

type RepoScore struct {
	Name       string  `json:"name"`
	Percentage float64 `json:"percentage"`
	Error      string  `json:"error"`
}

type expectedScoreReturn struct {
	Checks       []RepoScore `json:"checks"`
	Files        int         `json:"files"`
	Issues       int         `json:"issues"`
	Repo         string      `json:"repo"`
	Average      float64     `json:"average"`
	Grade        string      `json:"grade"`
	ResolvedRepo string      `json:"resolvedRepo"`
	LastRefresh  time.Time   `json:"last_refresh"`
}

func main() {
	reposFileDir := flag.String("repos", "", "A file containg all important repos")
	serverPath := flag.String("server", "", "The url to server like localhost:8000")
	outputFilename := flag.String("output", "sca_report.csv", "Name of the file to output result to.")
	flag.Parse()

	if reposFileDir == nil || serverPath == nil {
		fmt.Println("Must provide a repo path and server path to run this script")
		return
	}

	var repos []string
	reposFile, err := ioutil.ReadFile(*reposFileDir)
	if err != nil {
		log.Println("Could not read file")
		log.Println(*reposFileDir)
		log.Println(err)
	}
	repoReader := bytes.NewReader(reposFile)
	scanner := bufio.NewScanner(repoReader)
	for scanner.Scan() {
		repos = append(repos, scanner.Text())
	}
	repoScores := refreshRepos(repos, *serverPath)
	columns := []string{"path", "org", "repo", "grade", "weighted average", "files", "issues"}
	checks := []string{"golint", "gofmt", "megacheck", "misspell", "deadcode", "ineffassign", "go_vet", "errcheck",
		"goimports"}
	records := [][]string{}
	records = append(records, append(columns, checks...))
	for _, repoScore := range repoScores {
		record := []string{}

		orgRepo := strings.TrimPrefix(repoScore.Repo, "code.justin.tv/")
		orgRepoSplit := strings.Split(orgRepo, "/")
		if len(orgRepoSplit) != 2 {
			fmt.Println("The following repo could not be split into org and repo", orgRepo)
			continue
		}
		meta := []string{
			orgRepo,
			orgRepoSplit[0],
			orgRepoSplit[1],
			repoScore.Grade,
			fmt.Sprintf("%f", repoScore.Average),
			fmt.Sprintf("%d", repoScore.Files),
			fmt.Sprintf("%d", repoScore.Issues),
		}

		record = append(record, meta...)
		nameToCheck := make(map[string]RepoScore)
		for _, check := range repoScore.Checks {
			nameToCheck[check.Name] = check
		}
		for _, c := range checks {
			check := nameToCheck[c]
			record = append(record, fmt.Sprintf("%f", check.Percentage))
		}
		records = append(records, record)
	}

	file, err := os.Create(*outputFilename)
	if err != nil {
		fmt.Println(err)
	}
	defer file.Close()

	w := csv.NewWriter(file)
	for _, record := range records {
		if err := w.Write(record); err != nil {
			fmt.Println("error writing record to csv", err)
		}
	}
	w.Flush()

	if err := w.Error(); err != nil {
		fmt.Println(err)
	}
}

func refreshRepos(repos []string, serverPath string) []expectedScoreReturn {
	var scores []expectedScoreReturn
	baseURL := serverPath + "/score/"
	for _, r := range repos {
		log.Println("Processing repo: ", r)
		refreshURL := baseURL + r + "?forceRefresh=true"
		resp, err := http.Get(refreshURL)
		if err != nil {
			log.Printf("Encountered error while trying to get report for %v", r)
			log.Println(err.Error())
			continue
		}

		if resp.StatusCode != http.StatusOK {
			log.Printf("Goreportcard encountered issue with repo")
			log.Println(refreshURL)
			log.Println("code", resp.StatusCode)
			log.Println(resp.Body)
			continue
		}

		score := expectedScoreReturn{}

		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			log.Printf("Could not read response body")
			log.Println(err.Error())
		}
		err = json.Unmarshal(body, &score)
		if err != nil {
			log.Printf("Could not unmarshal body into json")
			log.Println(err.Error())
		}

		scores = append(scores, score)
		break
	}
	return scores
}
