package queue

import (
	"errors"
	"os"
	"path/filepath"
	"strings"

	"github.com/go-ini/ini"
	uuid "github.com/satori/go.uuid"
)

const (
	sectionHeader          = "sandstorm-agent"
	queueArnKeyName        = "QueueArn"
	queueURLKeyName        = "QueueURL"
	maxQueuePrefixLength   = 42
	maxNamePrefixLength    = 20
	defaultTopicArn        = "arn:aws:sns:us-west-2:734326455073:sandstorm-production"
	defaultQueueConfigPath = "/var/run/sandstorm/"
	defaultQueueFileName   = "queue_arn"
	defaultQueueNamePrefix = "sandstorm"
	defaultRegion          = "us-west-2"
)

//Config for Queue
type Config struct {
	QueueArn        string
	QueueURL        string
	TopicArn        string
	QueueConfigPath string
	QueueFileName   string
	QueueNamePrefix string
	WaitTimeSeconds int
}

func truncatePrefix(prefix string, maxLength int) (truncated string) {
	truncated = prefix
	if len(truncated) > maxLength {
		truncated = truncated[0:maxLength]
	}
	return
}

// buildQueueIdentifier returns a newly generated UUID and a FQDN
// Queuenames have a maximum length of 80 chars and and we prefix with 'sandstorm'
// and suffix wth a UUID. Hostnames are truncated down to 43-len(prefix) chars to fit
// name length
func buildQueueIdentifier(namePrefix, hostname string) (name string) {
	namePrefix = truncatePrefix(namePrefix, maxNamePrefixLength)
	hostname = truncatePrefix(hostname, maxQueuePrefixLength-len(namePrefix))

	uid := uuid.NewV4().String()
	name = strings.Join([]string{namePrefix, hostname, uid}, "-")
	name = strings.Replace(name, ".", "_", -1)
	return
}

// DefaultConfig creates a config with default values
func DefaultConfig() Config {

	return Config{
		TopicArn:        defaultTopicArn,
		QueueConfigPath: defaultQueueConfigPath,
		QueueFileName:   defaultQueueFileName,
		QueueNamePrefix: defaultQueueNamePrefix,
	}
}

// replaces nil values in provided config with defaultConfig
func mergeConfig(provided *Config, defaultConfig Config) {
	if provided.QueueConfigPath == "" {
		provided.QueueConfigPath = defaultConfig.QueueConfigPath
	}

	if provided.QueueFileName == "" {
		provided.QueueFileName = defaultConfig.QueueFileName
	}

	if provided.QueueNamePrefix == "" {
		provided.QueueNamePrefix = defaultConfig.QueueNamePrefix
	}

	if provided.TopicArn == "" {
		provided.TopicArn = defaultConfig.TopicArn
	}
}

//LoadQueueConfig loads queue config data from file,
func LoadQueueConfig(queueConfigPath string) (queueArn string, queueURL string, err error) {
	fh, err := ini.Load(queueConfigPath)
	if err != nil {
		return "", "", errors.New("error loading agent queue config: " + err.Error())
	}

	section := fh.Section(sectionHeader)
	arn, err := section.GetKey(queueArnKeyName)
	if err != nil {
		return "", "", errors.New("error reading queueArn from config file: " + err.Error())
	}
	url, err := section.GetKey(queueURLKeyName)
	if err != nil {
		return "", "", errors.New("error reading queueURL from config file: " + err.Error())
	}
	return arn.String(), url.String(), nil
}

func (cfg *Config) queueConfigPath() string {
	return filepath.Join(cfg.QueueConfigPath, cfg.QueueFileName)
}

//WriteQueueConfig writes the queue arn and queue url to a file on disk
// located at config.queueConfigPath(queueConfigPath())
func (cfg *Config) WriteQueueConfig() (err error) {
	f, err := os.Create(cfg.queueConfigPath())
	if err != nil {
		return
	}

	defer func() {
		closeErr := f.Close()
		if err != nil {
			err = errors.New("error closing queue config file: " + closeErr.Error())
		}
	}()

	iniFile := ini.Empty()
	section, err := iniFile.NewSection(sectionHeader)
	if err != nil {
		return
	}

	_, err = section.NewKey(queueArnKeyName, cfg.QueueArn)
	if err != nil {
		return
	}
	_, err = section.NewKey(queueURLKeyName, cfg.QueueURL)
	if err != nil {
		return
	}

	_, err = iniFile.WriteTo(f)
	if err != nil {
		return
	}

	err = f.Sync()
	return
}
