package api

import (
	"fmt"
	"net/http"
	"path"

	"goji.io"

	"goji.io/pat"

	"code.justin.tv/infosec/cors"
	"code.justin.tv/systems/guardian/cfg"
	glogrus "code.justin.tv/systems/guardian/glogrus2"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/derekdowling/go-json-spec-handler/jsh-api"
	"github.com/jixwanwang/apiutils"
	"github.com/sirupsen/logrus"
)

const (
	publicFolder      = "public"
	authorizationView = "/authorize"
)

// Logger is the logger used by the api
var Logger *logrus.Logger

// Build returns the fully constructed web handler with middleware
func Build(config *cfg.Config, db guardian.Storer, identifier guardian.Identifier, logger *logrus.Logger, stats statsd.Statter, changelog cfg.ChangelogClient) *goji.Mux {
	Logger = logger
	jshapi.SendHandler = jshapi.DefaultSender(Logger)

	guardianAPI, err := GuardianAPI(config, db, identifier, changelog)
	if err != nil {
		logger.Fatalf("Error creating Guardian API: %s", err.Error())
	}

	ldapAPI := LDAPAPI(config, identifier)

	router := goji.NewMux()

	router.Use(config.CORS.Policy().MustMiddleware)
	router.Use(cors.BlockOnOptions)

	// GET /authorize HTML Page and dependent asset serving
	router.Handle(pat.Get("/assets/*"), Assets(config.Assets.AssetPath))
	router.Handle(pat.Get("/favicon.ico"), Assets(config.Assets.AssetPath))
	router.Handle(pat.Get("/authorize"), Authorize(config.Assets.AssetPath))
	router.Handle(pat.Get("/health"), HealthCheck(db))
	router.Handle(pat.New(fmt.Sprintf("/%s/*", LDAPPrefix)), ldapAPI)

	router.Handle(pat.New(fmt.Sprintf("/%s/*", OAuthPrefix)), NewOAuth2Router(db, identifier, stats))

	// since this is a catchall, ensure this is registered with the router last otherwise
	// it will match other calls
	router.Handle(pat.New("/*"), guardianAPI)

	router.Use(glogrus.NewGlogrus(logger, "guardian"))

	return router
}

// GuardianAPI builds a new OAuth2 Guardian powered API
func GuardianAPI(config *cfg.Config, db guardian.Storer, identifier guardian.Identifier, c cfg.ChangelogClient) (*jshapi.API, error) {

	guardianAPI := jshapi.New("")

	auth, err := NewAuthMiddleware(config.Admin, db, identifier)
	if err != nil {
		return nil, err
	}

	guardianAPI.Use(auth.Middleware)

	//	build /clients* routes
	clientStorage := &ClientStorage{db, c}
	clientResource := jshapi.NewCRUDResource(clientType, clientStorage)
	clientResource.Action("reset", clientStorage.Reset)
	clientResource.Action("reset_secret", clientStorage.ResetSecret)

	guardianAPI.Add(clientResource)

	return guardianAPI, nil
}

// LDAPAPI builds API endpoints for querying LDAP resources
func LDAPAPI(config *cfg.Config, identifier guardian.Identifier) *jshapi.API {

	// build ldap API
	ldapAPI := jshapi.New(LDAPPrefix)

	ldapStorage := &IdentityStorage{Identifier: identifier}
	userResource := jshapi.NewResource("users")
	userResource.Get(ldapStorage.GetUser)
	userResource.List(ldapStorage.ListUsers)

	groupResource := jshapi.NewResource("groups")
	groupResource.Get(ldapStorage.GetGroup)
	groupResource.List(ldapStorage.ListGroups)

	ldapAPI.Add(groupResource)
	ldapAPI.Add(userResource)

	return ldapAPI
}

// Assets serves public asset files for the given folder path
func Assets(assetPath string) http.Handler {
	folderPath := path.Join(assetPath, publicFolder)
	return http.FileServer(http.Dir(folderPath))
}

// Authorize serves the new client authorization view that OAuth users must proceed
// through before gaining access to a client
func Authorize(assetPath string) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		contentPath := path.Join(assetPath, publicFolder, authorizationView+".html")
		http.ServeFile(w, r, contentPath)
	})
}

// HealthCheck checks authorizations and clients tables
func HealthCheck(db guardian.Storer) (handler http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		client, err := storage.TestClient(db)
		if err != nil {
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			Logger.Println(err)
			return
		}
		defer func() {
			err = db.DeleteClientByID(adminUser, client.GetID())
			if err != nil {
				Logger.Println(err)
			}
		}()

		accessData, err := storage.CreateTestAccessData(client)
		if err != nil {
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			Logger.Println(err)
		}
		err = db.SaveAccess(accessData)
		if err != nil {
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			Logger.Println(err)
			return
		}
		defer func() {
			err = db.RemoveAccess(accessData.AccessToken)
			if err != nil {
				Logger.Println(err)
			}
		}()

		_, err = db.LoadAccess(accessData.AccessToken)
		if err != nil {
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			Logger.Println(err)
			return
		}
		w.WriteHeader(http.StatusOK)
		return
	})
}
