package api

import (
	"context"
	"fmt"
	"net/http"

	"github.com/derekdowling/go-json-spec-handler"
	"github.com/derekdowling/go-json-spec-handler/jsh-api"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"

	"golang.org/x/oauth2"
)

// admin client attributes
const (
	adminClientName             = "guardian oauth clients management api"
	adminClientDescription      = adminClientName
	adminClientRedirectEndpoint = "/auth/callback"
)

type contextKey string

var (
	userContextKey contextKey = "user"
)

// AuthMiddleware provides handlers and middleware for authentication via oauth
type AuthMiddleware struct {
	oauthConfig *oauth2.Config
	db          guardian.Storer
	identifier  guardian.Identifier
	cfg         *cfg.AdminConfig
}

// NewAuthMiddleware returns a fully configured AuthHandler
func NewAuthMiddleware(cfg *cfg.AdminConfig, db guardian.Storer, identifier guardian.Identifier) (*AuthMiddleware, error) {
	redirectURI, err := cfg.RedirectURI(adminClientRedirectEndpoint)
	if err != nil {
		return nil, err
	}

	return &AuthMiddleware{
		oauthConfig: &oauth2.Config{
			ClientID:    cfg.ClientsID,
			RedirectURL: redirectURI,
		},
		db:         db,
		identifier: identifier,
		cfg:        cfg,
	}, nil
}

// create admin client
func (ah *AuthMiddleware) createAdminClient() (client *guardian.Client, err error) {
	client, err = guardian.NewClient(adminClientName,
		ah.oauthConfig.RedirectURL,
		adminClientDescription,
		ah.cfg.BaseURL)
	if err != nil {
		err = fmt.Errorf("error creating admin client: %s", err.Error())
		return
	}

	client.ID = ah.oauthConfig.ClientID
	err = ah.db.SaveClient(client)
	if err != nil {
		err = fmt.Errorf("error saving admin client: %s", err.Error())
		return
	}
	return
}

// getOrCreateAdminClient retrieves a configured admin client or creates a new one
func (ah *AuthMiddleware) getOrCreateAdminClient() error {
	c, err := ah.db.GetClient(ah.oauthConfig.ClientID)
	if err != nil {
		return err
	}

	if _, ok := c.(*guardian.Client); !ok {
		_, err = ah.createAdminClient()
		if err != nil {
			return err
		}
	}

	return nil
}

func (ah *AuthMiddleware) handleOAuthRequest(w http.ResponseWriter, r *http.Request) (req *http.Request) {
	req = r
	tokenStr := getToken(r)
	if tokenStr == "" {
		jshapi.SendHandler(w, r, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "Did not provide an OAuth Token",
			Status: http.StatusUnauthorized,
		})
		return
	}

	tc, valid := ah.db.CheckToken(tokenStr)
	if !valid || tc == nil || tc.User == nil {
		jshapi.SendHandler(w, r, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "Invalid Authorization token",
			Status: http.StatusUnauthorized,
		})
		return
	}

	req = r.WithContext(context.WithValue(r.Context(), userContextKey, tc.User))
	return
}

// Allows basic Auth requests to allow work. Useful for CLI or Postman type uses.
func (ah *AuthMiddleware) handleBasicAuthRequest(w http.ResponseWriter, r *http.Request) (req *http.Request) {
	req = r
	username, password, isBasicAuth := r.BasicAuth()
	if !isBasicAuth {
		return
	}

	user, err := ah.identifier.Authenticate(username, password)
	if err != nil || user == nil {
		jshapi.SendHandler(w, r, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "Invalid Basic Auth username/password combo.",
			Status: http.StatusUnauthorized,
		})
		return
	}

	// if all was successful, add the authorized user to the context
	req = r.WithContext(context.WithValue(r.Context(), userContextKey, user))
	return
}

// Middleware is a middleware handler that attempts to authenticate a user either
// via Basic Auth or through provider OAuth credentials.
// Authorization HTTP Header and 'token' query param and sets User and Group
// in context.
func (ah *AuthMiddleware) Middleware(inner http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {

		// if a user context was created via Basic Auth, skip OAuth verification
		r = ah.handleBasicAuthRequest(w, r)
		if r.Context().Value(userContextKey) != nil {
			inner.ServeHTTP(w, r)
			return
		}

		// otherwise try getting a user context from provided OAuth credentials
		r = ah.handleOAuthRequest(w, r)
		if r.Context().Value(userContextKey) != nil {
			inner.ServeHTTP(w, r)
		}
	})
}
