package api

import (
	"errors"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/mocks"
	"code.justin.tv/systems/guardian/guardian/storage"

	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/derekdowling/go-json-spec-handler"
	"github.com/derekdowling/go-json-spec-handler/client"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
)

func TestAuthMiddleware(t *testing.T) {
	db := storage.CreateTestDB()
	mockedIdentifier := new(mocks.Identifier)

	config := loadTestConfig()
	config.Changelog.CallerName = "guardian-testing"

	logger := logrus.New()

	changelog, err := cfg.ConfigureChangelog(config.Changelog, logger)
	if err != nil {
		logger.Fatal(err.Error())
	}

	api := Build(config, db, mockedIdentifier, logger, new(statsd.NoopClient), changelog)

	server := httptest.NewServer(api)
	defer server.Close()
	baseURL := server.URL

	const (
		testUsername = "testUserName"
		testPassword = "testPassword"
	)

	t.Run("authenticated", func(t *testing.T) {
		request, err := jsc.ListRequest(baseURL, "clients")
		assert.NoError(t, err)

		user := &guardian.User{
			UID: testUsername,
			Groups: []string{
				"team-something",
			},
		}
		request.SetBasicAuth(testUsername, testPassword)

		mockedIdentifier.On("Authenticate", testUsername, testPassword).Return(user, nil).Once()
		doc, resp, err := jsc.Do(request, jsh.ListMode)
		assert.Equal(t, http.StatusOK, resp.StatusCode)
		assert.NoError(t, err)
		assert.Empty(t, doc.Data)
	})

	t.Run("unauthenticated", func(t *testing.T) {
		request, err := jsc.ListRequest(baseURL, "clients")
		assert.NoError(t, err)

		request.SetBasicAuth(testUsername, testPassword)
		mockedIdentifier.On("Authenticate", testUsername, testPassword).Return(nil, errors.New("")).Once()

		_, resp, err := jsc.Do(request, jsh.ListMode)
		assert.Equal(t, http.StatusUnauthorized, resp.StatusCode)
		assert.NoError(t, err)
	})
}
