package api

import (
	"context"
	"fmt"
	"net/http"
	"strings"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"
	"code.justin.tv/systems/guardian/guardian/tokens"

	"github.com/derekdowling/go-json-spec-handler"
	uuid "github.com/satori/go.uuid"
)

const clientType = "clients"

// ClientStorage implements jshapi.Storage
type ClientStorage struct {
	db        guardian.Storer
	changelog cfg.ChangelogClient
}

// Save assigns a URL of 1 to the object
func (c *ClientStorage) Save(ctx context.Context, object *jsh.Object) (*jsh.Object, jsh.ErrorType) {
	client := &guardian.Client{}
	unmarshalErr := object.Unmarshal(clientType, client)
	if unmarshalErr != nil {
		return nil, unmarshalErr
	}

	client.ID = uuid.NewV4().String()
	client.HashAlgorithm = tokens.DefaultHashAlgorithm
	client.Scopes = guardian.DefaultClientScopes

	err := client.RegenerateSecret()
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error creating client: %s", err.Error()))
	}

	saveErr := c.db.SaveClient(client)
	if saveErr != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error saving client: %s", saveErr.Error()))
	}

	object.ID = client.ID

	marshalErr := object.Marshal(client)
	if marshalErr != nil {
		return nil, marshalErr
	}

	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		Logger.Error("No guardian user found")
	}

	description := fmt.Sprintf("Created client %s", client.Name)
	command := "ClientStorage.Save"
	additionalData := fmt.Sprintf("username=%s,client=%s", callerIdentity.CN, client.Name)

	c.changelog.LogInfoChangeEvent(client.Name, description, callerIdentity.CN, command, additionalData)

	return object, nil
}

// Get returns a resource with ID as specified by the request
func (c *ClientStorage) Get(ctx context.Context, id string) (*jsh.Object, jsh.ErrorType) {
	client, err := c.db.GetClient(id)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error getting client: %s", err.Error()))
	}
	if client == nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("No such client with ID: %s", id),
			Status: http.StatusNotFound,
		}
	}

	return jsh.NewObject(id, clientType, client)
}

// List returns a list of clients
func (c *ClientStorage) List(ctx context.Context) (jsh.List, jsh.ErrorType) {
	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "no user identity in request context",
			Status: http.StatusUnauthorized,
		}
	}

	clients, err := c.db.ListAuthorizedClients(callerIdentity, "", 0)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error fetching clients: %s", err.Error()))
	}

	list := jsh.List{}
	for _, client := range clients {
		object, err := jsh.NewObject(client.GetID(), clientType, client)
		if err != nil {
			return nil, err
		}

		list = append(list, object)
	}

	return list, nil
}

// Update saves changes to a client object
func (c *ClientStorage) Update(ctx context.Context, object *jsh.Object) (*jsh.Object, jsh.ErrorType) {
	clientPatch := &guardian.Client{}
	unmarshalErr := object.Unmarshal(clientType, clientPatch)
	if unmarshalErr != nil {
		return nil, unmarshalErr
	}

	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "no user identity in request context",
			Status: http.StatusUnauthorized,
		}
	}

	existingClient, fetchErr := c.db.GetAuthorizedClient(callerIdentity, object.ID)
	if fetchErr != nil {
		_, ok := fetchErr.(storage.UnauthorizedError)
		if ok {
			return nil, &jsh.Error{
				Title:  "Unauthorized",
				Detail: fetchErr.Error(),
				Status: http.StatusUnauthorized,
			}
		}
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("Could not find client for ID '%s'", object.ID),
			Status: http.StatusNotFound,
			ISE:    fetchErr.Error(),
		}
	}

	if existingClient == nil {
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "user does not have permission to update the client",
			Status: http.StatusUnauthorized,
		}
	}

	existingGuardianClient, ok := existingClient.(*guardian.Client)
	if !ok {
		return nil, jsh.ISE("existing client type mismatch")
	}

	oldGuardianClient := existingGuardianClient

	existingGuardianClient.RedirectURI = clientPatch.RedirectURI
	existingGuardianClient.Groups = clientPatch.Groups
	existingGuardianClient.AdminGroups = clientPatch.AdminGroups
	existingGuardianClient.Description = clientPatch.Description
	existingGuardianClient.Homepage = clientPatch.Homepage
	existingGuardianClient.Name = clientPatch.Name

	existingClient = existingGuardianClient

	saveErr := c.db.SaveClient(existingClient)
	if saveErr != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error saving client: %s", saveErr.Error()))
	}

	err := object.Marshal(existingClient)
	if err != nil {
		return nil, err
	}

	//send separate changelog event for each changeable field
	description := fmt.Sprintf("Updated client %s", clientPatch.Name)
	command := "ClientStorage.Update"
	if oldGuardianClient.RedirectURI != existingGuardianClient.RedirectURI {
		additionalData := fmt.Sprintf("username=%s,client=%s,oldRedirectURI=%s,newRedirectURI=%s",
			callerIdentity.CN, clientPatch.Name, oldGuardianClient.RedirectURI, existingGuardianClient.RedirectURI)

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}
	if compareGroups(oldGuardianClient.Groups, existingGuardianClient.Groups) {
		additionalData := fmt.Sprintf("username=%s,client=%s,OldGroups=%s,NewGroups=%s",
			callerIdentity.CN, clientPatch.Name, strings.Join(oldGuardianClient.Groups, ","),
			strings.Join(existingGuardianClient.Groups, ","))

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}
	if compareGroups(oldGuardianClient.AdminGroups, existingGuardianClient.AdminGroups) {
		additionalData := fmt.Sprintf("username=%s,client=%s,OldAdminGroups=%s,NewAdminGroups=%s",
			callerIdentity.CN, clientPatch.Name, strings.Join(oldGuardianClient.AdminGroups, ","),
			strings.Join(existingGuardianClient.AdminGroups, ","))

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}
	if oldGuardianClient.Description != existingGuardianClient.Description {
		additionalData := fmt.Sprintf("username=%s,client=%s,Description=%s,Description=%s",
			callerIdentity.CN, clientPatch.Name, oldGuardianClient.Description,
			existingGuardianClient.Description)

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}
	if oldGuardianClient.Homepage != existingGuardianClient.Homepage {
		additionalData := fmt.Sprintf("username=%s,client=%s,oldHomepage=%s,newHomepage=%s",
			callerIdentity.CN, clientPatch.Name, oldGuardianClient.Homepage,
			existingGuardianClient.Homepage)

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}
	if oldGuardianClient.Name != existingGuardianClient.Name {
		additionalData := fmt.Sprintf("username=%s,client=%s,oldName=%s,newName=%s",
			callerIdentity.CN, clientPatch.Name, oldGuardianClient.Name,
			existingGuardianClient.Name)

		c.changelog.LogInfoChangeEvent(clientPatch.Name, description, callerIdentity.CN, command, additionalData)
	}

	return object, nil
}

func compareGroups(oldGroups []string, newGroups []string) bool {
	var oldGroupsSet = make(map[string]struct{})
	for _, oldGroup := range oldGroups {
		oldGroupsSet[oldGroup] = struct{}{}
	}

	var newGroupsSet = make(map[string]struct{})
	for _, newGroup := range newGroups {
		newGroupsSet[newGroup] = struct{}{}
	}

	for _, newGroup := range newGroups {
		if _, ok := oldGroupsSet[newGroup]; !ok {
			return false
		}
	}

	for _, oldGroup := range oldGroups {
		if _, ok := newGroupsSet[oldGroup]; !ok {
			return false
		}
	}

	return true

}

// Delete removes a client
func (c *ClientStorage) Delete(ctx context.Context, id string) jsh.ErrorType {
	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		return &jsh.Error{
			Title:  "Unauthorized",
			Detail: "no user identity in request context",
			Status: http.StatusUnauthorized,
		}
	}

	guardianClient, fetchErr := c.db.GetAuthorizedClient(callerIdentity, id)
	if fetchErr != nil {
		_, ok := fetchErr.(storage.UnauthorizedError)
		if ok {
			return &jsh.Error{
				Title:  "Unauthorized",
				Detail: fetchErr.Error(),
				Status: http.StatusUnauthorized,
			}
		}
		return &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("Could not find client for ID '%s'", id),
			Status: http.StatusNotFound,
			ISE:    fetchErr.Error(),
		}
	}

	client, ok := guardianClient.(*guardian.Client)
	if !ok {
		return jsh.ISE("existing client type mismatch")
	}

	err := c.db.DeleteClientByID(callerIdentity, id)
	if err != nil {
		return jsh.ISE(fmt.Sprintf("Error deleting client: %s", err.Error()))
	}

	//send changelog event
	description := fmt.Sprintf("Deleted client %s", client.Name)
	command := "ClientStorage.Delete"
	additionalData := fmt.Sprintf("username=%s,client=%s,clientID=%s,Name=%s",
		callerIdentity.CN, client.Name, id, client.Name)

	c.changelog.LogInfoChangeEvent(client.Name, description, callerIdentity.CN, command, additionalData)

	return nil
}

// Reset invalidates all current authorization tokens for a given client
func (c *ClientStorage) Reset(ctx context.Context, id string) (*jsh.Object, jsh.ErrorType) {
	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "no user identity in request context",
			Status: http.StatusUnauthorized,
		}
	}

	guardianClient, fetchErr := c.db.GetAuthorizedClient(callerIdentity, id)
	if fetchErr != nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("Could not find client for ID '%s'", id),
			Status: http.StatusNotFound,
			ISE:    fetchErr.Error(),
		}
	}

	client, ok := guardianClient.(*guardian.Client)
	if !ok {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("Could not find client for ID '%s'", id),
			Status: http.StatusNotFound,
			ISE:    "client is nil",
		}
	}

	if client == nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("No such client with ID: %s", id),
			Status: http.StatusNotFound,
		}
	}

	client.NewVersion()
	err := c.db.SaveClient(client)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Unable to save client: %s", err.Error()))
	}

	object, jshErr := jsh.NewObject(client.GetID(), clientType, client)
	if jshErr != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error build jsh Object from client: %s", jshErr.Error()))
	}

	//send changelog event
	description := fmt.Sprintf("Reset all auth tokens for client %s", client.Name)
	command := "ClientStorage.Reset"
	additionalData := fmt.Sprintf("username=%s,client=%s,clientID=%s,Name=%s",
		callerIdentity.CN, client.Name, id, client.Name)

	c.changelog.LogInfoChangeEvent(client.Name, description, callerIdentity.CN, command, additionalData)

	return object, nil
}

// ResetSecret rotates a client's secret token. Will cause all existing OAuth tokens
// to be invalidated.
func (c *ClientStorage) ResetSecret(ctx context.Context, id string) (*jsh.Object, jsh.ErrorType) {
	callerIdentity, ok := ctx.Value(userContextKey).(*guardian.User)
	if !ok {
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: "no user identity in request context",
			Status: http.StatusUnauthorized,
		}
	}

	guardianClient, fetchErr := c.db.GetAuthorizedClient(callerIdentity, id)
	if fetchErr != nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("Could not find client for ID '%s'", id),
			Status: http.StatusNotFound,
			ISE:    fetchErr.Error(),
		}
	}

	client, ok := guardianClient.(*guardian.Client)
	if !ok {
		return nil, jsh.ISE("existing client type mismatch")
	}

	if client == nil {
		return nil, jsh.NotFound(clientType, id)
	}

	err := client.RegenerateSecret()
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Unable to regenerate client secret: %s", err.Error()))
	}

	err = c.db.SaveClient(client)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Unable to save client: %s", err.Error()))
	}

	obj, jshErr := jsh.NewObject(id, clientType, client)
	if jshErr != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error build jsh Object from client: %s", jshErr.Error()))
	}

	//send changelog event
	description := fmt.Sprintf("Rotated secret token for client %s", client.Name)
	command := "ClientStorage.ResetSecret"
	additionalData := fmt.Sprintf("username=%s,client=%s,clientID=%s,Name=%s",
		callerIdentity.CN, client.Name, id, client.Name)

	c.changelog.LogInfoChangeEvent(client.Name, description, callerIdentity.CN, command, additionalData)

	return obj, nil
}
