package api

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/mocks"
	"code.justin.tv/systems/guardian/guardian/storage"

	"code.justin.tv/systems/guardian/cfg"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/derekdowling/go-json-spec-handler"
	"github.com/derekdowling/go-json-spec-handler/client"
	"github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
)

func TestClientAPI(t *testing.T) {

	db := storage.CreateTestDB()
	mockedIdentifier := new(mocks.Identifier)
	db.Identifier = mockedIdentifier

	config := loadTestConfig()
	config.Changelog.CallerName = "guardian-testing"

	logger := logrus.New()

	changelog, err := cfg.ConfigureChangelog(config.Changelog, logger)
	if err != nil {
		logger.Fatal(err.Error())
	}

	api := Build(config, db, mockedIdentifier, logger, new(statsd.NoopClient), changelog)

	server := httptest.NewServer(api)
	defer server.Close()
	baseURL := server.URL

	validToken, err := newTestToken(db)
	if err != nil {
		t.Fatal(err)
	}

	adminToken, err := newTestSSEToken(db)
	if err != nil {
		t.Fatal(err)
	}

	defer func() {
		err := db.RemoveAccess(validToken.AccessToken)
		if err != nil {
			t.Log(err)
		}
		err = db.RemoveAccess(adminToken.AccessToken)
		if err != nil {
			t.Log(err)
		}
	}()

	t.Run("DELETE /clients:id", func(t *testing.T) {
		client, err := storage.TestClient(db)
		if client != nil {
			defer db.DeleteClientByID(adminUser, client.ID)
		}
		if err != nil {
			t.Fatal(err)
		}

		t.Run("unauthorized", func(t *testing.T) {
			mockedIdentifier.On("GetUser", regularUserCN).Return(regularUser, nil).Once()

			req, err := jsc.DeleteRequest(baseURL, clientType, client.GetID())
			assert.NoError(t, err)
			validToken.SetAuthHeader(req)

			_, resp, err := jsc.Do(req, jsh.ObjectMode)
			assert.NoError(t, err)
			assert.Equal(t, http.StatusUnauthorized, resp.StatusCode)

			retrievedClient, err := db.GetClient(client.GetID())
			assert.NoError(t, err)
			client.Secret = ""
			assert.Equal(t, client, retrievedClient)

			mockedIdentifier.AssertExpectations(t)
		})

		t.Run("authorized", func(t *testing.T) {
			mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()

			req, err := jsc.DeleteRequest(baseURL, clientType, client.GetID())
			assert.NoError(t, err)
			adminToken.SetAuthHeader(req)

			_, resp, err := jsc.Do(req, jsh.ObjectMode)
			assert.NoError(t, err)
			assert.Equal(t, http.StatusNoContent, resp.StatusCode)
			mockedIdentifier.AssertExpectations(t)
		})
	})

	t.Run("GET /clients", func(t *testing.T) {
		const testGroup = "test-group"
		testClients := make([]*guardian.Client, 2)
		for i := range testClients {
			testClient, err := storage.TestClient(db)
			if err != nil {
				t.Fatal(err)
			}
			if i == 1 {
				testClient.AdminGroups = []string{testGroup}
				err = db.SaveClient(testClient)
				if err != nil {
					t.Fatal(err)
				}
			}
			testClients[i] = testClient
		}
		defer func(clients []*guardian.Client) {
			for _, object := range clients {
				if object != nil {
					db.DeleteClientByID(adminUser, object.ID)
				}
			}
		}(testClients)

		t.Run("should successfully return clients with admin token", func(t *testing.T) {
			mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()
			defer mockedIdentifier.AssertExpectations(t)

			request, err := jsc.ListRequest(baseURL, "clients")
			assert.NoError(t, err)

			adminToken.SetAuthHeader(request)

			doc, resp, err := jsc.Do(request, jsh.ListMode)
			assert.Equal(t, http.StatusOK, resp.StatusCode)
			assert.NoError(t, err)

			list := doc.Data
			assert.NotEmpty(t, list)

			returnedClientIDs := make([]string, len(list))
			for i, v := range list {
				returnedClientIDs[i] = v.ID
			}

			for _, c := range testClients {
				assert.Contains(t, returnedClientIDs, c.GetID())
			}
		})

		t.Run("should successfully return clients with non-admin token", func(t *testing.T) {
			mockedIdentifier.On("GetUser", regularUserCN).Return(&guardian.User{Groups: []string{testGroup}}, nil).Once()
			defer mockedIdentifier.AssertExpectations(t)

			request, err := jsc.ListRequest(baseURL, "clients")
			assert.NoError(t, err)

			validToken.SetAuthHeader(request)

			doc, resp, err := jsc.Do(request, jsh.ListMode)
			assert.Equal(t, http.StatusOK, resp.StatusCode)
			assert.NoError(t, err)

			list := doc.Data
			assert.NotEmpty(t, list)

			returnedClientIDs := make([]string, len(list))
			for i, v := range list {
				returnedClientIDs[i] = v.ID
			}
			assert.Len(t, returnedClientIDs, 1)

			assert.Condition(t, func() bool {
				var matches int
				for _, rit := range returnedClientIDs {
					for _, c := range testClients {
						if rit == c.GetID() {
							matches++
						}
					}
				}
				if matches == len(returnedClientIDs) {
					return true
				}
				return false
			})
		})
	})

	Convey("Admin API Tests", t, func() {
		Convey("POST /clients", func() {

			client, err := storage.NewTestClient()
			So(err, ShouldBeNil)
			obj, objErr := jsh.NewObject("", clientType, client)
			So(objErr, ShouldBeNil)

			Convey("should create a new client", func() {
				mockedIdentifier.On("GetUser", regularUserCN).Return(regularUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				req, newReqErr := jsc.PostRequest(baseURL, obj)
				So(newReqErr, ShouldBeNil)
				validToken.SetAuthHeader(req)

				doc, resp, err := jsc.Do(req, jsh.ObjectMode)
				So(resp.StatusCode, ShouldEqual, http.StatusCreated)
				So(err, ShouldBeNil)

				object := doc.First()

				createdClient, getErr := db.GetClient(object.ID)
				So(getErr, ShouldBeNil)
				So(createdClient, ShouldNotBeNil)

				deleteErr := db.DeleteClientByID(adminUser, object.ID)
				So(deleteErr, ShouldBeNil)
			})
		})

		Convey("POST /clients with admin groups should store admin groups in the db.", func() {
			client, err := storage.NewTestClient()
			client.AdminGroups = []string{"test-group", "team-sse"}
			So(err, ShouldBeNil)
			obj, objErr := jsh.NewObject("", clientType, client)
			So(objErr, ShouldBeNil)

			mockedIdentifier.On("GetUser", regularUserCN).Return(regularUser, nil).Once()
			defer mockedIdentifier.AssertExpectations(t)

			req, newReqErr := jsc.PostRequest(baseURL, obj)
			So(newReqErr, ShouldBeNil)
			validToken.SetAuthHeader(req)
			doc, resp, err := jsc.Do(req, jsh.ObjectMode)
			So(resp.StatusCode, ShouldEqual, http.StatusCreated)
			So(err, ShouldBeNil)

			object := doc.First()
			defer func() {
				err := db.DeleteClientByID(adminUser, object.ID)
				if err != nil {
					t.Log("error deleting client")
				}
			}()

			createdClient, getErr := db.GetClient(object.ID)
			So(getErr, ShouldBeNil)
			So(createdClient, ShouldNotBeNil)

			So(createdClient.(*guardian.Client).AdminGroups, ShouldResemble, client.AdminGroups)

			deleteErr := db.DeleteClientByID(adminUser, object.ID)
			So(deleteErr, ShouldBeNil)

		})

		Convey("GET /clients/:id", func() {

			client, err := storage.TestClient(db)
			if client != nil {
				defer db.DeleteClientByID(adminUser, client.ID)
			}
			So(err, ShouldBeNil)

			Convey("should successfully return a client", func() {
				mockedIdentifier.On("GetUser", regularUserCN).Return(regularUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				request, err := jsc.FetchRequest(baseURL, "clients", client.GetID())
				So(err, ShouldBeNil)

				validToken.SetAuthHeader(request)

				doc, resp, err := jsc.Do(request, jsh.ObjectMode)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)
				So(err, ShouldBeNil)
				object := doc.First()
				So(object.ID, ShouldEqual, client.GetID())

				respClient := &guardian.Client{}
				unmarshalErr := object.Unmarshal(clientType, respClient)
				So(unmarshalErr, ShouldBeNil)

				matchClient := new(guardian.Client)
				*matchClient = *client
				matchClient.Secret = ""
				So(respClient, ShouldResemble, matchClient)
			})
		})

		Convey("PATCH /clients/:id", func() {

			client, err := storage.TestClient(db)
			if client != nil {
				defer db.DeleteClientByID(adminUser, client.ID)
			}
			So(err, ShouldBeNil)

			Convey("should succeed and only overwrite updated values", func() {
				mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				partialClient := &guardian.Client{
					ID:          client.GetID(),
					Description: "patched description",
					Groups:      []string{"team-syseng"},
					RedirectURI: client.RedirectURI,
					Name:        client.Name,
				}

				patchObj, err := jsh.NewObject(partialClient.GetID(), clientType, partialClient)
				So(err, ShouldBeNil)

				req, newReqErr := jsc.PatchRequest(baseURL, patchObj)
				So(newReqErr, ShouldBeNil)
				adminToken.SetAuthHeader(req)

				doc, resp, doErr := jsc.Do(req, jsh.ObjectMode)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)
				So(doErr, ShouldBeNil)
				object := doc.First()

				respClient := &guardian.Client{}
				unmarshalErr := object.Unmarshal(clientType, respClient)
				So(unmarshalErr, ShouldBeNil)

				// check that the patch worked properly, and empty values
				// didn't override previous ones
				So(respClient.GetID(), ShouldEqual, client.GetID())
				So(respClient.Description, ShouldEqual, partialClient.Description)
				So(respClient.Groups, ShouldResemble, partialClient.Groups)
				So(respClient.RedirectURI, ShouldEqual, client.RedirectURI)
				So(respClient.Name, ShouldEqual, client.Name)
			})

			Convey("should fail if user is not in admin groups", func() {
				mockedIdentifier.On("GetUser", regularUserCN).Return(regularUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				partialClient := &guardian.Client{
					ID:          client.GetID(),
					Description: "patched description",
					Groups:      []string{guardian.TeamSSELDAPGroup},
				}

				patchObj, err := jsh.NewObject(partialClient.GetID(), clientType, partialClient)
				So(err, ShouldBeNil)

				req, newReqErr := jsc.PatchRequest(baseURL, patchObj)
				So(newReqErr, ShouldBeNil)
				validToken.SetAuthHeader(req)

				_, resp, doErr := jsc.Do(req, jsh.ObjectMode)
				So(resp.StatusCode, ShouldEqual, http.StatusUnauthorized)
				So(doErr, ShouldBeNil)

			})

			Convey("should be able to set groups to empty list", func() {
				mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)
				client.AdminGroups = []string{guardian.TeamSSELDAPGroup, "test-group-1"}
				client.Groups = []string{guardian.TeamSSELDAPGroup, "test-group-1"}
				db.SaveClient(client)

				partialClient := &guardian.Client{
					ID:          client.GetID(),
					Description: "patched description",
					Groups:      []string{},
				}
				patchObj, err := jsh.NewObject(partialClient.GetID(), clientType, partialClient)
				So(err, ShouldBeNil)

				req, newReqErr := jsc.PatchRequest(baseURL, patchObj)
				So(newReqErr, ShouldBeNil)
				adminToken.SetAuthHeader(req)

				_, resp, doErr := jsc.Do(req, jsh.ObjectMode)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)
				So(doErr, ShouldBeNil)

				c, getErr := db.GetClient(partialClient.ID)
				So(getErr, ShouldBeNil)
				So(c.(*guardian.Client).Groups, ShouldBeEmpty)
				So(c.(*guardian.Client).AdminGroups, ShouldResemble, []string{guardian.TeamSSELDAPGroup})
			})
		})

		Convey("GET /clients/:id/reset", func() {
			client, err := storage.TestClient(db)
			if client != nil {
				defer db.DeleteClientByID(adminUser, client.ID)
			}
			So(err, ShouldBeNil)

			Convey("should reset client version", func() {
				mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				req, newReqErr := jsc.ActionRequest(baseURL, clientType, client.ID, "reset")
				So(newReqErr, ShouldBeNil)
				adminToken.SetAuthHeader(req)

				doc, response, err := jsc.Do(req, jsh.ObjectMode)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(err, ShouldBeNil)

				c, getErr := db.GetClient(client.ID)
				So(getErr, ShouldBeNil)
				So(c.GetVersion(), ShouldNotEqual, client.GetVersion())

				respClient := &guardian.Client{}
				unmarshalErr := doc.First().Unmarshal(clientType, respClient)
				So(unmarshalErr, ShouldBeNil)

				So(respClient, ShouldResemble, c)
			})
		})

		Convey("GET /clients/:id/reset_secret", func() {
			client, err := storage.TestClient(db)
			if client != nil {
				defer db.DeleteClientByID(adminUser, client.ID)
			}
			So(err, ShouldBeNil)

			Convey("should reset client version and secret", func() {
				mockedIdentifier.On("GetUser", adminUserCN).Return(adminUser, nil).Once()
				defer mockedIdentifier.AssertExpectations(t)

				req, newReqErr := jsc.ActionRequest(baseURL, clientType, client.ID, "reset_secret")
				So(newReqErr, ShouldBeNil)
				adminToken.SetAuthHeader(req)

				doc, response, err := jsc.Do(req, jsh.ObjectMode)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(err, ShouldBeNil)

				obj := doc.First()

				respClient := &guardian.Client{}
				marshalErr := json.Unmarshal(obj.Attributes, respClient)
				So(marshalErr, ShouldBeNil)
				So(respClient.Secret, ShouldNotEqual, client.Secret)
				So(respClient.Secret, ShouldNotBeEmpty)

				c, getErr := db.GetClient(client.ID)
				So(getErr, ShouldBeNil)
				So(c.GetSecretHash(), ShouldNotEqual, client.GetSecretHash())
			})
		})
	})
}

func TestClientUtil(t *testing.T) {
	Convey("->compareGroups", t, func() {
		testGroupA := []string{"a", "b", "c"}
		testGroupB := []string{"d", "e", "f"}

		Convey("should return true when groups are the same", func() {
			So(compareGroups(testGroupA, testGroupA), ShouldBeTrue)
		})
		Convey("should return false when groups are different", func() {
			So(compareGroups(testGroupA, testGroupB), ShouldBeFalse)
		})
	})
}
