package api

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/systems/guardian/guardian"
	jsh "github.com/derekdowling/go-json-spec-handler"
	"github.com/derekdowling/go-json-spec-handler/jsh-api"
)

// routes
const (
	LDAPPrefix = "ldap"
	GroupType  = "groups"
	UserType   = "users"
)

// IdentityStorage is a JSHAPI resource regarding fetching LDAP user/group info
type IdentityStorage struct {
	Identifier guardian.Identifier
}

// NewIdentityStorage creates an object with LDAP API route handlers
func NewIdentityStorage(identifier guardian.Identifier) *jshapi.API {

	resource := &IdentityStorage{Identifier: identifier}

	api := jshapi.New("")

	users := jshapi.NewResource(UserType)
	users.Get(resource.GetUser)
	users.List(resource.ListUsers)

	groups := jshapi.NewResource(GroupType)
	groups.Get(resource.GetGroup)
	groups.List(resource.ListGroups)

	api.Add(users)
	api.Add(groups)

	return api
}

// GetUser is a handler for GET /resource/users/:uid
func (res *IdentityStorage) GetUser(ctx context.Context, id string) (*jsh.Object, jsh.ErrorType) {
	user, err := res.Identifier.GetUserByName(id)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error getting user: %s", err.Error()))
	}

	if user == nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("No such user with uid: %s", id),
			Status: http.StatusNotFound,
		}
	}
	return jsh.NewObject(id, UserType, user)
}

// ListUsers is a handler for GET /resource/users
func (res *IdentityStorage) ListUsers(ctx context.Context) (jsh.List, jsh.ErrorType) {
	users, err := res.Identifier.ListUsers()
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error getting users: %s", err.Error()))
	}
	list := make(jsh.List, len(users))

	for i, user := range users {
		object, err := jsh.NewObject(user.CN, UserType, user)
		if err != nil {
			return nil, err
		}
		list[i] = object
	}
	return list, nil
}

// ListGroups is a handler for GET /resource/groups/:group_cn
func (res *IdentityStorage) ListGroups(ctx context.Context) (jsh.List, jsh.ErrorType) {
	groups, err := res.Identifier.ListGroups()
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error getting groups: %s", err.Error()))
	}

	list := make(jsh.List, len(groups))

	for i, group := range groups {

		object, err := jsh.NewObject(group.CN, GroupType, group)
		if err != nil {
			return nil, err
		}

		list[i] = object
	}

	return list, nil
}

// GetGroup is a handler for GET /resource/groups
func (res *IdentityStorage) GetGroup(ctx context.Context, id string) (*jsh.Object, jsh.ErrorType) {
	group, err := res.Identifier.GetGroup(id)
	if err != nil {
		return nil, jsh.ISE(fmt.Sprintf("Error getting user: %s", err.Error()))
	}

	if group == nil {
		return nil, &jsh.Error{
			Title:  "Not Found",
			Detail: fmt.Sprintf("No such group with CN: %s", id),
			Status: http.StatusNotFound,
		}
	}

	return jsh.NewObject(id, GroupType, group)
}
