package api

import (
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"

	"net/http"
	"net/http/httptest"

	"strings"
	"testing"

	"code.justin.tv/systems/guardian/cfg"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/derekdowling/go-json-spec-handler/client"
	uuid "github.com/satori/go.uuid"
	"github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

func TestIdentityResource(t *testing.T) {

	Convey("Identity Resource Endpoint Tests", t, func() {

		db := storage.CreateTestDB()
		testIdentifier := &guardian.TestIdentifier{}

		config := loadTestConfig()
		config.Changelog.CallerName = "guardian-testing"

		logger := logrus.New()

		changelog, err := cfg.ConfigureChangelog(config.Changelog, logger)
		if err != nil {
			logger.Fatal(err.Error())
		}

		api := Build(config, db, testIdentifier, logger, new(statsd.NoopClient), changelog)

		server := httptest.NewServer(api)
		defer server.Close()

		baseURL := strings.Join([]string{server.URL, LDAPPrefix}, "/")

		Convey("For Users", func() {

			Convey("Getting all Users", func() {
				doc, response, err := jsc.List(baseURL, UserType)

				So(err, ShouldBeNil)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(doc, ShouldNotBeNil)
			})

			Convey("Getting a Single User", func() {

				doc, response, err := jsc.Fetch(baseURL, UserType, "guardian")
				So(err, ShouldBeNil)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(doc, ShouldNotBeNil)

				expectedUser, err := testIdentifier.GetUserByName("guardian")
				So(err, ShouldBeNil)

				returnedUser := &guardian.User{}
				err = doc.First().Unmarshal(UserType, returnedUser)
				So(err, ShouldBeNil)

				So(expectedUser.CN, ShouldEqual, returnedUser.CN)
				So(expectedUser.UID, ShouldEqual, returnedUser.UID)
				So(expectedUser.GIDNumber, ShouldEqual, returnedUser.GIDNumber)
				So(expectedUser.Email, ShouldEqual, returnedUser.Email)

			})

			Convey("Getting a nonexistent user", func() {
				_, response, err := jsc.Fetch(baseURL, UserType, uuid.NewV4().String())
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.StatusCode, ShouldEqual, http.StatusNotFound)
			})
		})
		Convey("For Groups", func() {

			Convey("Getting all Groups", func() {
				doc, response, err := jsc.List(baseURL, GroupType)

				So(err, ShouldBeNil)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(doc, ShouldNotBeNil)
			})

			Convey("Getting a Single Group", func() {
				doc, response, err := jsc.Fetch(baseURL, GroupType, "navi")

				So(err, ShouldBeNil)
				So(response.StatusCode, ShouldEqual, http.StatusOK)
				So(doc, ShouldNotBeNil)

				expectedGroup, err := testIdentifier.GetGroup("navi")
				So(err, ShouldBeNil)

				returnedGroup := &guardian.Group{}
				err = doc.First().Unmarshal(GroupType, returnedGroup)
				So(err, ShouldBeNil)

				So(expectedGroup.GID, ShouldEqual, returnedGroup.GID)
				So(len(expectedGroup.Members), ShouldEqual, len(returnedGroup.Members))

				//assuming 1 member
				So(expectedGroup.Members[0], ShouldEqual, returnedGroup.Members[0])

			})
		})
	})
}
