package api

import (
	"fmt"
	"net/http"
	"strings"

	"goji.io/pat"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"
	"code.justin.tv/systems/guardian/osin"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/derekdowling/go-json-spec-handler"
	"github.com/derekdowling/go-json-spec-handler/jsh-api"
	"github.com/jixwanwang/apiutils"
	"goji.io"
)

// routes
const (
	OAuthPrefix        = "oauth2"
	AuthorizeEndpoint  = "/authorize"
	TokenEndpoint      = "/token"
	CheckTokenEndpoint = "/check_token"
)

const (
	invalidTokenBucket = "check_token.invalid_token"
	successTokenBucket = "check_token.success"
	ldapFailBucket     = "ldap.failure"
)

// OAuth2Router provides handlers for oauth2 authorization flow
type OAuth2Router struct {
	db         guardian.Storer
	osin       *osin.Server
	identifier guardian.Identifier
	Stats      statsd.Statter
}

// NewOAuth2Router returns a new router implementing oauth routes
func NewOAuth2Router(db guardian.Storer, identifier guardian.Identifier, statter statsd.Statter) *goji.Mux {
	osinConfig := osin.NewServerConfig()
	osinConfig.AllowGetAccessRequest = true
	osinConfig.AllowClientSecretInParams = true
	osinConfig.ErrorStatusCode = http.StatusBadRequest
	osinConfig.AllowedAccessTypes = osin.AllowedAccessType{osin.AUTHORIZATION_CODE, osin.CLIENT_CREDENTIALS}
	osinConfig.RedirectURISeparator = ","

	osinServer := osin.NewServer(osinConfig, db)
	osinServer.AccessTokenGen = storage.DefaultTokenGenerator
	osinServer.AuthorizeTokenGen = storage.DefaultTokenGenerator

	oar := &OAuth2Router{
		db:         db,
		osin:       osinServer,
		identifier: identifier,
		Stats:      statter,
	}

	mux := goji.SubMux()
	mux.HandleFunc(pat.Get(AuthorizeEndpoint), oar.Authorize)
	mux.HandleFunc(pat.Post(AuthorizeEndpoint), oar.Authorize)

	mux.HandleFunc(pat.Get(TokenEndpoint), oar.Token)
	mux.HandleFunc(pat.Post(TokenEndpoint), oar.Token)

	mux.HandleFunc(pat.Get(CheckTokenEndpoint), oar.CheckToken)

	return mux
}

// serveJSON serves v as json object with correct headers for oauth endpoints
func serveJSON(w http.ResponseWriter, v interface{}) {
	osin.AddHeaders(w)
	apiutils.ServeJSON(w, v)
	return
}

// CheckToken checks a token for validity
func (oar *OAuth2Router) CheckToken(w http.ResponseWriter, r *http.Request) {
	token := getToken(r)
	var err error
	if token == "" {
		err = oar.Stats.Inc(invalidTokenBucket, 1, 1)
		if err != nil {
			Logger.Errorf("failed to increment %s: %s", invalidTokenBucket, err.Error())
			err = nil
		}
		jshapi.SendHandler(w, r, &jsh.Error{
			Title:  "Bad Request",
			Detail: "token required via basic auth or query param",
			Status: http.StatusBadRequest,
		})
		return
	}

	response, ok := oar.db.CheckToken(token)
	if !ok {
		err = oar.Stats.Inc(invalidTokenBucket, 1, 1)
		if err != nil {
			Logger.Errorf("failed to increment %s: %s", invalidTokenBucket, err.Error())
			err = nil
		}
		jshapi.SendHandler(w, r, &jsh.Error{
			Title:  "Not Found",
			Detail: "token not found",
			Status: http.StatusNotFound,
		})
		return
	}

	err = oar.Stats.Inc(successTokenBucket, 1, 1)
	if err != nil {
		Logger.Errorf("failed to increment %s: %s", successTokenBucket, err.Error())
		err = nil
	}
	serveJSON(w, response)
	return
}

// Authorize implements oauth2 authorization endpoint
func (oar *OAuth2Router) Authorize(w http.ResponseWriter, r *http.Request) {
	err := r.ParseForm()
	if err != nil {
		// show login page here
		jshapi.SendHandler(w, r, jsh.ISE(fmt.Sprintf("Error parsing POST form: %s", err.Error())))
		return
	}

	formValues := r.Form

	if formValues.Get("client_id") == "" {
		jshapi.SendHandler(w, r, jsh.InputError("Missing client_id", "client_id"))
		return
	}

	user, authError := oar.UserFromRequest(r)
	if authError != nil {
		jshapi.SendHandler(w, r, authError)
		return
	}

	resp := oar.osin.NewResponse()
	defer resp.Close()

	if authRequest := oar.osin.HandleAuthorizeRequest(resp, r); authRequest != nil {
		client, ok := authRequest.Client.(*guardian.Client)
		if !ok || client == nil {
			jshapi.SendHandler(w, r, jsh.ISE("error loading client: type mismatch"))
			return
		}

		authRequest.Authorized = client.AuthorizeUser(user)
		authRequest.UserData = user

		oar.osin.FinishAuthorizeRequest(resp, r, authRequest)
	}

	if resp.IsError && resp.InternalError != nil {
		jshapi.SendHandler(w, r, handleOsinError(resp, "error handling authorization request: "))
		return
	}

	err = osin.OutputJSON(resp, w, r)
	if err != nil {
		jshapi.SendHandler(w, r, jsh.ISE(err.Error()))
		return
	}
}

// UserFromRequest checks the various methods for getting user credentials and checks
// to see if they are both a valid user, and that their credentials are correct. If so,
// returns the corresponding user.
func (oar *OAuth2Router) UserFromRequest(r *http.Request) (*guardian.User, *jsh.Error) {
	// try basic auth first, default back to checking an input form for credentials
	username, password, hasAuth := r.BasicAuth()
	if !hasAuth {

		formValues := r.Form
		username = formValues.Get("username")
		password = formValues.Get("password")

		// see if there are any Form credentials
		if username == "" || password == "" {
			// show login page here
			return nil, &jsh.Error{
				Title:  "Unauthorized",
				Detail: "Missing authentication credentials",
				Status: http.StatusUnauthorized,
			}
		}
	}

	user, err := oar.identifier.Authenticate(username, password)
	if err != nil {
		statsErr := oar.Stats.Inc(ldapFailBucket, 1, 1)
		if statsErr != nil {
			Logger.Errorf("failed to increment %s: %s", ldapFailBucket, err.Error())
		}
		return nil, &jsh.Error{
			Title:  "Unauthorized",
			Detail: fmt.Sprintf("User credentials not valid: %s", err.Error()),
			Status: http.StatusUnauthorized,
		}
	}

	return user, nil
}

// Token implements oauth2 token endpoint, supports GET/POST requests and is used to
// exchange an authorization code for a valid OAuth Token
func (oar *OAuth2Router) Token(w http.ResponseWriter, r *http.Request) {
	resp := oar.osin.NewResponse()
	defer resp.Close()

	if ar := oar.osin.HandleAccessRequest(resp, r); ar != nil {
		ar.Authorized = true
		oar.osin.FinishAccessRequest(resp, r, ar)
	}

	if resp.IsError && resp.InternalError != nil {
		jshapi.SendHandler(w, r, handleOsinError(resp, "error handling token request: "))
		return
	}

	err := osin.OutputJSON(resp, w, r)
	if err != nil {
		jshapi.SendHandler(w, r, jsh.ISE(err.Error()))
		return
	}
}

// getToken retrieves token from request query parameters and from basic auth.
// Token retrieved from query params is prioritized over basic auth.
func getToken(r *http.Request) (token string) {
	token = r.URL.Query().Get("token")
	if token != "" {
		return
	}

	_, token, _ = r.BasicAuth()
	if token != "" {
		return
	}

	authHeader := strings.Split(r.Header.Get("Authorization"), " ")
	if len(authHeader) == 2 && strings.EqualFold(authHeader[0], "bearer") {
		if authHeader[1] != "" {
			token = authHeader[1]
			return
		}
	}

	return
}

func handleOsinError(r *osin.Response, prefix string) (jshErr *jsh.Error) {
	if r == nil || r.InternalError == nil || !r.IsError {
		return nil
	}

	msg := fmt.Sprintf("%s: %s", osin.NewDefaultErrors().Get(r.ErrorId), r.InternalError.Error())
	var statusCode int
	switch r.ErrorId {
	case osin.E_INVALID_REQUEST, osin.E_INVALID_SCOPE, osin.E_INVALID_GRANT, osin.E_UNSUPPORTED_GRANT_TYPE, osin.E_UNSUPPORTED_RESPONSE_TYPE:
		statusCode = http.StatusBadRequest
	case osin.E_ACCESS_DENIED, osin.E_UNAUTHORIZED_CLIENT:
		statusCode = http.StatusUnauthorized
	case osin.E_TEMPORARILY_UNAVAILABLE:
		statusCode = http.StatusServiceUnavailable
	case osin.E_SERVER_ERROR:
		return jsh.ISE(r.InternalError.Error())
	}

	return &jsh.Error{
		Title:  http.StatusText(statusCode),
		Detail: msg,
		Status: statusCode,
	}
}
