package api

import (
	"encoding/json"
	"log"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strings"
	"testing"

	"golang.org/x/oauth2"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"
	"code.justin.tv/systems/guardian/osin"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

func TestOAuth(t *testing.T) {

	db := storage.CreateTestDB()
	resource := &guardian.TestIdentifier{}

	config := loadTestConfig()
	config.Changelog.CallerName = "guardian-testing"

	logger := logrus.New()

	changelog, err := cfg.ConfigureChangelog(config.Changelog, logger)
	if err != nil {
		logger.Fatal(err.Error())
	}

	api := Build(config, db, resource, logger, new(statsd.NoopClient), changelog)
	server := httptest.NewServer(api)
	defer server.Close()

	baseURL := strings.Join([]string{server.URL, OAuthPrefix}, "/")

	tokenEndpoint, err := getEndpointURL(baseURL, TokenEndpoint)
	if err != nil {
		log.Fatal(err.Error())
	}

	authorizeEndpoint, err := getEndpointURL(baseURL, AuthorizeEndpoint)
	if err != nil {
		log.Fatal(err.Error())
	}

	checkTokenEndpoint, err := getEndpointURL(baseURL, CheckTokenEndpoint)
	if err != nil {
		log.Fatal(err.Error())
	}

	Convey("OAuth Endpoint tests", t, func() {

		client, err := storage.TestClient(db)
		if client != nil {
			defer db.DeleteClientByID(adminUser, client.GetID())
		}
		So(err, ShouldBeNil)

		oauthConfig := &oauth2.Config{
			ClientID:     client.ID,
			ClientSecret: client.Secret,
			Scopes:       []string{"scope1", "scope2"},
			Endpoint: oauth2.Endpoint{
				AuthURL:  authorizeEndpoint.String(),
				TokenURL: tokenEndpoint.String(),
			},
		}

		Convey("GET /authorize", func() {

			var (
				authCode string
				token    *oauth2.Token
			)

			tr := &http.Transport{}

			const (
				state    = "something"
				scope    = "scope1,scope2"
				username = "guardian"
				password = "kappa"
			)

			data := url.Values{}
			data.Set("redirect_uri", client.GetRedirectURI())
			data.Set("state", state)
			data.Set("scope", scope)
			data.Set("client_id", client.GetID())
			data.Set("response_type", string(osin.CODE))

			Convey("via basic auth", func() {

				requestURL := authorizeEndpoint
				requestURL.RawQuery = data.Encode()

				request, requestErr := http.NewRequest("GET", requestURL.String(), nil)
				So(requestErr, ShouldBeNil)

				request.SetBasicAuth(username, password)

				// use RoundTrip to intercept the redirect, using http.Get would
				// instantly follow it
				resp, getErr := tr.RoundTrip(request)
				So(getErr, ShouldBeNil)
				defer resp.Body.Close()

				So(resp.StatusCode, ShouldEqual, http.StatusFound)

				redirectURL, parseErr := url.Parse(resp.Header.Get("Location"))
				So(parseErr, ShouldBeNil)

				values := redirectURL.Query()
				So(values.Get("code"), ShouldNotBeEmpty)
				So(values.Get("state"), ShouldNotBeEmpty)

				Convey("GET /token + /check_token", func() {

					authCode = values.Get("code")

					var exchangeErr error

					// implicitly calls /token
					token, exchangeErr = oauthConfig.Exchange(oauth2.NoContext, authCode)
					So(exchangeErr, ShouldBeNil)
					So(token, ShouldNotBeEmpty)
					So(token.Valid(), ShouldBeTrue)
					oauthClient := oauthConfig.Client(oauth2.NoContext, token)
					resp, checkTokenErr := oauthClient.Get(checkTokenEndpoint.String())
					So(checkTokenErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					checkTokenResponse := &guardian.TokenCheck{}
					decodeErr := json.NewDecoder(resp.Body).Decode(checkTokenResponse)
					So(decodeErr, ShouldBeNil)
					So(checkTokenResponse.Token.Valid(), ShouldBeTrue)

					testResource := &guardian.TestIdentifier{}
					testUser, getUserErr := testResource.GetUserByName(checkTokenResponse.User.UID)
					So(getUserErr, ShouldBeNil)
					So(checkTokenResponse.User, ShouldResemble, testUser)

				})
			})

			Convey("via POST form", func() {

				data.Set("username", username)
				data.Set("password", password)

				requestURL := authorizeEndpoint
				requestURL.RawQuery = data.Encode()

				// use RoundTrip to intercept the redirect, using http.Get would
				// instantly follow it
				request, requestErr := http.NewRequest("GET", requestURL.String(), nil)
				So(requestErr, ShouldBeNil)

				// use RoundTrip to intercept the redirect, using http.Get would
				// instantly follow it
				resp, getErr := tr.RoundTrip(request)
				So(getErr, ShouldBeNil)
				defer resp.Body.Close()

				So(resp.StatusCode, ShouldEqual, http.StatusFound)

				redirectURL, parseErr := url.Parse(resp.Header.Get("Location"))
				So(parseErr, ShouldBeNil)

				values := redirectURL.Query()
				So(values.Get("code"), ShouldNotBeEmpty)
				So(values.Get("state"), ShouldNotBeEmpty)
			})

		})
	})
}
