package api

import (
	"github.com/cactus/go-statsd-client/statsd"
	"goji.io/pat"
	"golang.org/x/oauth2"

	"goji.io"

	"net/http"
	"net/url"
	"path"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/storage"
)

var adminUser = &guardian.User{
	Groups: []string{guardian.TeamSSELDAPGroup},
}

var regularUser = &guardian.User{}

const (
	localDynamoEndpoint = "http://localhost:9000"
)

// LoadTestConfig Helper function to easily load test config
func loadTestConfig() *cfg.Config {
	config := cfg.DefaultConfig()

	config.Debug = true

	return config
}

func getEndpointURL(baseURL string, relpath string) (*url.URL, error) {
	endpoint, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}

	endpoint.Path = path.Join(endpoint.Path, relpath)
	return endpoint, nil
}

func testOAuthEndpoint(db *storage.Storage, identifier guardian.Identifier) (mux *goji.Mux) {
	mux = goji.NewMux()
	mux.Handle(pat.New(OAuthPrefix+"*"), NewOAuth2Router(db, identifier, new(statsd.NoopClient)))
	return
}

func testAuthenticationMux(validToken string) *goji.Mux {
	mux := goji.NewMux()
	mux.Handle(pat.Get("/oauth2/check_token"), http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		tokenStr := getToken(r)
		if tokenStr != validToken {
		}
	}))
	return mux
}

func newTestSSEToken(db *storage.Storage) (token *oauth2.Token, err error) {
	return newTestAccessToken(db, true)
}

const (
	regularUserCN = "regular user"
	adminUserCN   = "admin user"
)

func newTestAccessToken(db *storage.Storage, admin bool) (token *oauth2.Token, err error) {
	client, err := storage.TestClient(db)
	if err != nil {
		return
	}
	accessData, err := storage.CreateTestAccessData(client)
	if err != nil {
		return
	}

	testUser := accessData.UserData.(*guardian.User)
	testUser.CN = regularUserCN
	if admin {
		testUser.CN = adminUserCN
	}
	accessData.UserData = testUser

	err = db.SaveAccess(accessData)
	if err != nil {
		return
	}
	token = &oauth2.Token{AccessToken: accessData.AccessToken}
	return
}

func newTestToken(db *storage.Storage) (token *oauth2.Token, err error) {
	return newTestAccessToken(db, false)
}
