package cfg

import (
	"code.justin.tv/systems/changelog-clients/go/ccg"
	"github.com/sirupsen/logrus"
)

// NoopClient implements a noop client
type noopClient struct {
}

// LogInfoChangeEvent sends the event to changelog
func (c *noopClient) LogInfoChangeEvent(target string, description string, username string, command string, additionalData string) {
}

// LogWarnChangeEvent sends the event to changelog
func (c *noopClient) LogWarnChangeEvent(target string, description string, username string, command string, additionalData string) {
}

// changelogClient is the changelog client
type changelogClient struct {
	*ccg.Client
	logger   logrus.FieldLogger
	category string
}

// ChangelogConfig holds configuration options for changelog
type ChangelogConfig struct {
	Enabled  bool   `yaml:"enabled"`
	Category string `yaml:"category"`
	Host     string `yaml:"host"`

	CallerName string `yaml:"caller_name"`
}

// ChangelogClient interface
type ChangelogClient interface {
	LogInfoChangeEvent(target string, description string, username string, command string, additionalData string)
	LogWarnChangeEvent(target string, description string, username string, command string, additionalData string)
}

// ConfigureChangelog configures the changelog client
func ConfigureChangelog(config *ChangelogConfig, logger logrus.FieldLogger) (client ChangelogClient, err error) {
	if !config.Enabled {
		logger.Info("changelog disabled")
		client = &noopClient{}
		return
	}

	authedClient, err := ccg.NewS2SClient(&ccg.S2SConfig{
		CallerName: config.CallerName,
		Host:       config.Host,
	}, logger)
	if err != nil {
		return
	}

	c := &changelogClient{
		Client:   authedClient,
		logger:   logger,
		category: DefaultChangelogCategory,
	}
	logger.Debug("changelog enabled")

	if config.Category != "" {
		c.category = config.Category
	}
	client = c
	return
}

// LogInfoChangeEvent sends the event to changelog
func (c *changelogClient) LogInfoChangeEvent(target string, description string, username string, command string, additionalData string) {
	c.logChangeEvent(target, description, username, command, additionalData, ccg.SevInfo)
}

// LogWarnChangeEvent sends the event to changelog
func (c *changelogClient) LogWarnChangeEvent(target string, description string, username string, command string, additionalData string) {
	c.logChangeEvent(target, description, username, command, additionalData, ccg.SevWarning)
}

func (c *changelogClient) logChangeEvent(target string, description string, username string, command string, additionalData string, severity int) {
	if c.Client == nil {
		return
	}

	changeEvent := &ccg.Event{
		Category:       c.category,
		Criticality:    severity,
		Target:         target,
		Description:    description,
		Username:       username,
		Command:        command,
		AdditionalData: additionalData,
	}
	err := c.SendEvent(changeEvent)
	if err != nil {
		c.logger.WithFields(logrus.Fields{
			"event": changeEvent,
		}).Errorf("couldn't send changelog entry: %s", err)
	}
}
