package cfg

import (
	"fmt"
	"net/url"
	"os"
	"path"
	"strings"
	"time"

	yaml "gopkg.in/yaml.v2"

	"code.justin.tv/infosec/cors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	awssession "github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/bsdlp/config"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/imdario/mergo"
	"github.com/sirupsen/logrus"
)

// configuration defaults
const (
	DefaultLDAPAddress             = "ldap-usw2.internal.justin.tv"
	DefaultLDAPPort                = 636
	DefaultLDAPUser                = "guardian"
	DefaultTestClientsTable        = "clients-testing"
	DefaultTestAuthorizationsTable = "authorizations-testing"
	DefaultTestRoleArn             = "arn:aws:iam::634047383455:role/guardian-service-testing"
	DefaultLogPrefix               = "guardian"
	AppName                        = "guardian"
	DefaultClientsID               = "c54dca37-1dda-4826-9eee-339bb0b3b777"
	DefaultDevBaseURL              = "http://localhost:8181"
	DefaultProdBaseURL             = "https://guardian.prod.us-west2.justin.tv/"
	DefaultDebug                   = false
	DefaultAWSRegion               = "us-west-2"
	StagingLDAPAddress             = "ldap.dev.us-west2.justin.tv"
	StagingLDAPPort                = 636
	StagingLDAPUser                = "Guardian User"
	StagingLDAPPassword            = "test_password"
	DefaultCORSMaxAge              = "5h"
	DefaultReportRate              = 5 * time.Second
	DefaultChangelogCategory       = "guardian-staging"
	DefaultChangelogHost           = "changelog-s2s-staging.internal.justin.tv"
	DefaultChangelogLogDir         = "/var/log/changelog_client"
	DefaultCallerName              = "guardian-staging"
)

// Config holds config options for guardian
type Config struct {
	LDAP      *LDAPConfig      `yaml:"ldap"`
	DB        *DBConfig        `yaml:"db"`
	Logging   *LoggingConfig   `yaml:"logging"`
	Admin     *AdminConfig     `yaml:"admin"`
	Assets    *AssetConfig     `yaml:"assets"`
	CORS      *CORSConfig      `yaml:"cors"`
	Metrics   MetricsConfig    `yaml:"metrics"`
	Debug     bool             `yaml:"debug"`
	Changelog *ChangelogConfig `yaml:"changelog"`
}

// GetLogger returns logger based on config options
func (c *Config) GetLogger() (logger *logrus.Logger, err error) {
	logger = logrus.New()

	logger.Level, err = logrus.ParseLevel(c.Logging.Level)
	if err != nil {
		err = fmt.Errorf("error parsing logging level: %s", err.Error())
		return
	}

	logger.Formatter = new(logrus.JSONFormatter)
	return
}

// MetricsConfig holds config options for metrics
type MetricsConfig struct {
	GraphiteHostPort string        `yaml:"graphite_host_port"`
	ReportRate       time.Duration `yaml:"report_rate"`
	Environment      string        `yaml:"environment"`
}

// ConfigureStatsd returns a statsd.Statter based on configuration options
func (m MetricsConfig) ConfigureStatsd() (stats statsd.Statter, err error) {
	hostName, err := os.Hostname()
	if err != nil {
		hostName = "unknown"
	}

	hostPrefix := AppName + "." + m.Environment + "." + hostName
	stats, err = statsd.NewBufferedClient(m.GraphiteHostPort, hostPrefix, 1*time.Second, 512)
	if err != nil {
		return
	}
	return
}

// AdminConfig holds config options for clients admin api
type AdminConfig struct {
	// ClientsID is the oauth2 client id for the clients admin api. Defaults to
	// c54dca37-1dda-4826-9eee-339bb0b3b777.
	ClientsID string `yaml:"clients_id"`
	// BaseURL is the base url for the admin endpoint. Defaults to
	// http://localhost:8181/.
	BaseURL string `yaml:"base_url"`
}

// RedirectURI constructs redirect uri for the clients admin api based on config
// values and provided relpath ("/auth/callback")
func (ac *AdminConfig) RedirectURI(relpath string) (uri string, err error) {
	rURI, err := url.Parse(ac.BaseURL)
	if err != nil {
		err = fmt.Errorf("error parsing configured base_url: %s", err.Error())
		return
	}
	rURI.Path = path.Join(rURI.Path, relpath)
	uri = rURI.String()
	return
}

// LoggingConfig holds config options for logging
type LoggingConfig struct {
	Level string `yaml:"level"`
}

// LDAPConfig holds config options for ldap
type LDAPConfig struct {
	UseTLS       bool   `yaml:"use_tls"`
	Address      string `yaml:"address"`
	Port         int    `yaml:"port"`
	BindUser     string `yaml:"bind_user"`
	BindPassword string `yaml:"bind_password"`
}

// DBConfig holds dynamodb config options
type DBConfig struct {
	ClientsTable        string `yaml:"clients_table"`
	AuthorizationsTable string `yaml:"authorizations_table"`
	AWSRegion           string `yaml:"aws_region"`
	AWSConfig           *aws.Config
	AWSSession          *awssession.Session
	RoleArn             string `yaml:"role_arn"`
}

// CORSConfig holds cors config options
type CORSConfig struct {
	AllowedOrigins []cors.OriginList `yaml:"allowed_origins"`
	// duration, in a format parseable by https://golang.org/pkg/time/#ParseDuration
	MaxAge string `yaml:"max_age"`
	maxAge time.Duration
}

// SetMaxAge parses max age and sets CORSConfig.maxAge
func (c *CORSConfig) SetMaxAge() (err error) {
	duration, err := time.ParseDuration(c.MaxAge)
	if err != nil {
		return
	}
	c.maxAge = duration
	return
}

// Policy returns a cors.Policy
func (c *CORSConfig) Policy() cors.Policy {
	return cors.Policy{
		MaxAge:         cors.MaxAge(c.maxAge),
		AllowedOrigins: cors.Origins(c.AllowedOrigins...),
		AllowHeaders:   cors.HeaderList("Authorization"),
	}
}

// AssetConfig manages asset paths and settings
type AssetConfig struct {
	AssetPath string `yaml:"asset_path"`
}

// BuildDefaultAssetPath returns an absolute path to the expect public asset folder
func BuildDefaultAssetPath() string {
	// this is specifically for making Godeps magic paths play nicely
	gopath := strings.Split(os.Getenv("GOPATH"), "src")[0]

	return path.Join(gopath, "/src/code.justin.tv/systems/guardian")
}

// DefaultConfig returns a default config object
func DefaultConfig() *Config {
	return &Config{
		LDAP: &LDAPConfig{
			UseTLS:   true,
			Address:  DefaultLDAPAddress,
			Port:     DefaultLDAPPort,
			BindUser: DefaultLDAPUser,
		},
		DB: &DBConfig{
			ClientsTable:        DefaultTestClientsTable,
			AuthorizationsTable: DefaultTestAuthorizationsTable,
			AWSRegion:           DefaultAWSRegion,
			AWSConfig:           awsConfig(DefaultAWSRegion, DefaultTestRoleArn),
		},
		Logging: &LoggingConfig{
			Level: "info",
		},
		Admin: &AdminConfig{
			ClientsID: DefaultClientsID,
			BaseURL:   DefaultDevBaseURL,
		},
		Assets: &AssetConfig{
			AssetPath: BuildDefaultAssetPath(),
		},
		CORS: &CORSConfig{
			AllowedOrigins: []cors.OriginList{
				"http://localhost",
				"https://localhost",
				"https://dashboard.internal.justin.tv",
				DefaultProdBaseURL,
			},
			MaxAge: DefaultCORSMaxAge,
		},
		Debug: DefaultDebug,
		Changelog: &ChangelogConfig{
			Enabled:    false,
			Category:   DefaultChangelogCategory,
			Host:       DefaultChangelogHost,
			CallerName: DefaultCallerName,
		},
	}
}

// UseStagingLDAP changes LDAP properties to represent staging environment
func (c *Config) UseStagingLDAP() {
	c.LDAP.Port = StagingLDAPPort
	c.LDAP.Address = StagingLDAPAddress
	c.LDAP.BindUser = StagingLDAPUser
	c.LDAP.BindPassword = StagingLDAPPassword
}

func awsConfig(region, roleArn string) (config *aws.Config) {
	config = aws.NewConfig().WithRegion(region)
	if roleArn != "" {
		stsclient := sts.New(awssession.New(config))
		arp := &stscreds.AssumeRoleProvider{
			Duration:     900 * time.Second,
			ExpiryWindow: 10 * time.Second,
			RoleARN:      roleArn,
			Client:       stsclient,
		}
		creds := credentials.NewCredentials(arp)
		config.WithCredentials(creds)
	}
	return
}

// LoadConfig reads config file from either user or system config
// paths:
// * $HOME/.config/twitch/guardian/config.yaml
// * /etc/twitch/guardian/config.yaml
func LoadConfig() (c *Config, err error) {
	c = &Config{}

	err = config.Config{
		Organization: "twitch",
		Service:      "guardian",
		FileFormat: &config.FileFormat{
			Unmarshaller: yaml.Unmarshal,
			Extension:    "yaml",
		},
	}.Load(c)
	if err != nil {
		if err == config.ErrConfigFileNotFound {
			err = nil
		} else {
			return
		}
	}

	err = mergo.Merge(c, DefaultConfig())
	if err != nil {
		return
	}

	// check if we can parse cors.MaxAge
	err = c.CORS.SetMaxAge()
	if err != nil {
		return
	}

	c.DB.AWSConfig = awsConfig(c.DB.AWSRegion, c.DB.RoleArn)
	c.DB.AWSSession = awssession.New(c.DB.AWSConfig)
	return
}
