package main

import (
	"log"
	"net/http"
	"os"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"

	"code.justin.tv/common/gometrics"
	"code.justin.tv/systems/guardian/api"
	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian/ldap"
	"code.justin.tv/systems/guardian/guardian/storage"
)

const defaultListenAddress = "0.0.0.0:8181"

func main() {
	config, err := cfg.LoadConfig()
	if err != nil {
		log.Fatalln(err)
	}

	logger, err := config.GetLogger()
	if err != nil {
		logger.Fatal(err.Error())
	}

	changelog, err := cfg.ConfigureChangelog(config.Changelog, logger)
	if err != nil {
		logger.Fatal(err.Error())
	}

	var stats statsd.Statter
	stats = new(statsd.NoopClient)
	if config.Metrics.GraphiteHostPort != "" {
		stats, err = config.Metrics.ConfigureStatsd()
		if err != nil {
			logger.WithFields(logrus.Fields{"error": err}).Warnf("error configuring runtime Statsd: %s", err.Error())
			stats = new(statsd.NoopClient)
			err = nil
		}
	}
	gometrics.Monitor(stats, time.Second*5)

	identifier := ldap.NewIdentifier(config)
	store := storage.NewDynamoDBStorage(config, identifier)

	stack := api.Build(config, store, identifier, logger, stats, changelog)

	logger.Printf("Server listening at: %s", defaultListenAddress)

	if len(os.Args) > 1 && os.Args[1] == "--local" {
		logger.Fatal(http.ListenAndServeTLS(defaultListenAddress, "./test/cert.pem", "./test/key.pem", stack))
	} else {
		logger.Fatal(http.ListenAndServe(defaultListenAddress, stack))
	}
}
