package ldap

import (
	"fmt"
	"testing"

	"code.justin.tv/systems/guardian/guardian"

	uuid "github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
)

func TestLDAP(t *testing.T) {
	Convey("LDAP Setup", t, func() {
		rt, err := newResourceTest()
		So(err, ShouldBeNil)

		// create map of groupcn: [array,of,users]
		groupToUsers := make(map[string][]string)
		groupToUsersMap(rt.Users.Enabled, groupToUsers)
		groupToUsersMap(rt.Users.Disabled, groupToUsers)

		Convey("ListGroups should return created groups", func() {
			groups, err := rt.Resource.ListGroups()
			So(err, ShouldBeNil)

			// create a set of groups, then test for existence
			allGroups := make(map[string]interface{})
			for _, group := range groups {
				allGroups[group.CN] = nil
			}

			var (
				errors []error
				ok     bool
			)
			for _, group := range rt.GroupsNeeded {
				if _, ok = allGroups[group.CN]; ok == false {
					errors = append(errors, fmt.Errorf("%s doesn't exist", group.CN))
				}
			}
			So(errors, ShouldBeEmpty)
		})
		Convey("GetGroup should find groups by cn, and have all the right users", func() {
			var (
				group              *guardian.Group
				LDAPGroup          *guardian.Group
				membersInLDAPGroup map[string]interface{}
				ok                 bool
			)

			for _, group = range rt.GroupsNeeded {
				LDAPGroup, err = rt.Resource.GetGroup(group.CN)
				jsonGroupMembers := groupToUsers[group.CN]

				// Create a 'set' of users
				membersInLDAPGroup = make(map[string]interface{})
				for _, ldapMember := range LDAPGroup.Members {
					membersInLDAPGroup[ldapMember] = nil
				}

				// should be equal
				for _, jsonMember := range jsonGroupMembers {
					_, ok = membersInLDAPGroup[jsonMember]

					So(ok, ShouldEqual, true)
				}

			}

		})
		Convey("Users should", func() {
			var (
				returnedUser *guardian.User
				err          error
			)
			Convey("Be found if searched by cn", func() {
				for _, user := range rt.Users.Enabled {
					returnedUser, err = rt.Resource.GetUser(user.CN)
					So(err, ShouldBeNil)

					So(returnedUser, ShouldNotBeNil)
					So(returnedUser.UID, ShouldEqual, user.UID)
				}
			})

			Convey("Be found if searched by uid", func() {
				for _, user := range rt.Users.Enabled {
					returnedUser, err = rt.Resource.GetUserByName(user.UID)
					So(err, ShouldBeNil)

					So(returnedUser.CN, ShouldEqual, user.CN)
				}
			})

			Convey("Return nil if does not exist", func() {
				returnedUser, err = rt.Resource.GetUserByName(uuid.NewV4().String())
				So(err, ShouldBeNil)
				So(returnedUser, ShouldBeNil)
			})

			Convey("Be able to Bind if Enabled", func() {
				for _, user := range rt.Users.Enabled {
					returnedUser, err = rt.Resource.Authenticate(user.UID, "navi rules!")
					So(err, ShouldBeNil)

					So(returnedUser.CN, ShouldEqual, user.CN)
				}
			})

			Convey("Shouldn't exist if Disabled", func() {
				for _, user := range rt.Users.Disabled {
					returnedUser, err = rt.Resource.Authenticate(user.UID, "navi rules!")
					So(err, ShouldNotBeNil)

					So(returnedUser, ShouldBeNil)
				}
			})
		})

	})

	Convey("escapeLDAPArgument", t, func() {
		Convey("should leave ok characters as-is", func() {
			response := escapeLDAPArgument("test user")
			So(response, ShouldEqual, "test user")
		})

		Convey("should escape ()", func() {
			response := escapeLDAPArgument("test user (user)")
			So(response, ShouldEqual, "test user \\28user\\29")
		})
	})
}
