package storage

import (
	"encoding/hex"
	"testing"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/tokens"

	. "github.com/smartystreets/goconvey/convey"
)

func TestAccessStorage(t *testing.T) {
	Convey("Access Data Storage Tests", t, func() {
		storage := CreateTestDB()

		client, err := TestClient(storage)
		if client != nil {
			defer storage.DeleteClientByID(adminUser, client.GetID())
		}
		So(client, ShouldNotBeNil)
		So(err, ShouldBeNil)
		saveClientErr := storage.SaveClient(client)
		So(saveClientErr, ShouldBeNil)

		accessData, err := CreateTestAccessData(client)
		So(err, ShouldBeNil)
		So(accessData, ShouldNotBeNil)

		Convey("->LoadAccess", func() {
			saveErr := storage.SaveAccess(accessData)
			So(saveErr, ShouldBeNil)

			data, loadErr := storage.LoadAccess(accessData.AccessToken)
			So(loadErr, ShouldBeNil)
			So(data, ShouldNotBeNil)
			So(data.AccessToken, ShouldBeEmpty)

			secretHash, _, hashErr := tokens.DefaultTokenGenerator.HashSecret(accessData.AccessToken, nil)
			So(hashErr, ShouldBeNil)

			// stored as seconds since epoch vs nanosecond resolution
			So(data.CreatedAt.Unix(), ShouldEqual, accessData.CreatedAt.Unix())
			data.CreatedAt = accessData.CreatedAt

			So(data.AccessToken, ShouldBeEmpty)
			accessData.AccessTokenHash = secretHash
			data.AccessToken = accessData.AccessToken

			strippedClient := &guardian.Client{}
			*strippedClient = *client
			strippedClient.Secret = ""
			accessData.Client = strippedClient
			So(data, ShouldResemble, accessData)
		})

		Convey("->DeleteAccess", func() {
			deleteErr := storage.RemoveAccess(accessData.AccessToken)
			So(deleteErr, ShouldBeNil)

			data, loadErr := storage.LoadAccess(accessData.AccessToken)
			So(loadErr, ShouldBeNil)
			So(data, ShouldBeNil)
		})

		Convey("->CheckToken", func() {
			Convey("invalid token", func() {
				invalidToken := hex.EncodeToString([]byte("invalid_token"))
				tc, valid := storage.CheckToken(invalidToken)
				So(valid, ShouldBeFalse)
				So(tc, ShouldBeNil)
			})

			Convey("valid token", func() {
				checkTokenData, err := CreateTestAccessData(client)
				if checkTokenData != nil {
					defer storage.RemoveAccess(checkTokenData.AccessToken)
				}
				So(checkTokenData, ShouldNotBeNil)
				So(err, ShouldBeNil)
				saveErr := storage.SaveAccess(checkTokenData)
				So(saveErr, ShouldBeNil)

				tc, valid := storage.CheckToken(checkTokenData.AccessToken)
				So(valid, ShouldBeTrue)
				So(tc, ShouldNotBeNil)
				So(tc.User, ShouldResemble, guardian.TestUser)

			})
		})
	})
}
