package storage

import (
	"errors"
	"fmt"
	"net/http"
	"reflect"
	"strconv"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/osin"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

type UnauthorizedError struct {
	User *guardian.User
}

func (ue UnauthorizedError) Error() string {
	return fmt.Sprintf("User %s is not authorized to perform this action", ue.User.CN)
}

func (ue UnauthorizedError) Code() int {
	return http.StatusUnauthorized
}

func ensureSSEAccessToClient(client *guardian.Client) {
	var exists bool
	for _, group := range client.AdminGroups {
		if group == guardian.TeamSSELDAPGroup {
			exists = true
		}
	}

	if !exists {
		client.AdminGroups = append(client.AdminGroups, guardian.TeamSSELDAPGroup)
	}
}

// SaveClient saves a client to dynamodb
func (db *Storage) SaveClient(client osin.Client) (err error) {
	c, ok := client.(*guardian.Client)
	if !ok {
		err = errors.New("guardian: client is not of type *guardian.Client")
	}
	ensureSSEAccessToClient(c)

	item, err := dynamodbattribute.ConvertToMap(*c)
	if err != nil {
		return
	}

	delete(item, "secret")

	params := &dynamodb.PutItemInput{
		TableName: aws.String(db.Config.ClientsTable),
		Item:      item,
	}
	_, err = db.DB.PutItem(params)
	if err != nil {
		return
	}
	return
}

// GetAuthorizedClient retrieves a client by ID from dynamodb and checks if a
// user has access to it.
func (db *Storage) GetAuthorizedClient(user *guardian.User, clientID string) (client osin.Client, err error) {
	gClient, err := db.getClient(clientID)
	if err != nil {
		return
	}
	if !gClient.AuthorizeAdminUser(user) {
		return nil, UnauthorizedError{
			User: user,
		}
	}

	client = gClient
	return
}

func (db *Storage) getClient(clientID string) (client *guardian.Client, err error) {
	if clientID == "" {
		err = errors.New("guardian: id required when getting client")
		return
	}

	params := &dynamodb.GetItemInput{
		TableName: aws.String(db.Config.ClientsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"id": {S: aws.String(clientID)},
		},
		ConsistentRead: aws.Bool(true),
	}

	output, err := db.DB.GetItem(params)
	if err != nil {
		err = fmt.Errorf("Error retrieving client from Dynamo: %s", err.Error())
		return
	}

	if len(output.Item) == 0 {
		return nil, nil
	}

	c := &guardian.Client{}
	err = dynamodbattribute.ConvertFromMap(output.Item, c)
	if err != nil {
		err = fmt.Errorf("Error building client from Dynamo: %s", err.Error())
		return
	}

	var emptyClient guardian.Client
	if !reflect.DeepEqual(*c, emptyClient) {
		client = c
	}
	return
}

// GetClient retrieves a client by ID from dynamodb
// If no item is found, return nil client
func (db *Storage) GetClient(id string) (client osin.Client, err error) {
	return db.getClient(id)
}

// DeleteClient deletes a client by ID from dynamodb
// NOT CALLED
func (db *Storage) DeleteClient(client osin.Client) (err error) {
	err = errors.New("DeleteClient is deprecated")
	return
}

// DeleteClientByID deletes the client from dynamodb
func (db *Storage) DeleteClientByID(user *guardian.User, id string) (err error) {
	if user == nil || len(user.Groups) == 0 {
		return
	}

	condition, expressionAttributeValues := prepareAdminFilter(user.Groups)
	params := &dynamodb.DeleteItemInput{
		ConditionExpression:       aws.String(condition),
		ExpressionAttributeValues: expressionAttributeValues,
		TableName:                 aws.String(db.Config.ClientsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"id": {S: aws.String(id)},
		},
	}
	_, err = db.DB.DeleteItem(params)
	return
}

func prepareAdminFilter(groups []string) (filterExpression string, expressionAttributeValues map[string]*dynamodb.AttributeValue) {
	if len(groups) == 0 {
		return
	}

	const filterExpressionFormatString = "contains(admin_groups, :%d)"
	filterExpression = fmt.Sprintf(filterExpressionFormatString, 0)

	expressionAttributeValues = map[string]*dynamodb.AttributeValue{
		":0": {S: aws.String(groups[0])},
	}

	if len(groups) == 1 {
		return
	}

	for index, group := range groups[1:] {
		filterExpression += " or " + fmt.Sprintf(filterExpressionFormatString, index+1)
		expressionAttributeValues[":"+strconv.Itoa(index+1)] = &dynamodb.AttributeValue{
			S: aws.String(group),
		}
	}
	return
}

func (db *Storage) listAuthorizedClients(user *guardian.User, startID string, limit int64) (clients []osin.Client, err error) {
	params := &dynamodb.ScanInput{
		TableName: aws.String(db.Config.ClientsTable),
	}

	// if user is not nil, check authorization by ldap groups
	if user != nil {
		// if user isn't in any groups, return empty list of clients
		if len(user.Groups) == 0 {
			return
		}

		filterExpression, expressionAttributeValues := prepareAdminFilter(user.Groups)
		params.FilterExpression = aws.String(filterExpression)
		params.ExpressionAttributeValues = expressionAttributeValues
	}

	if limit != 0 {
		params.Limit = aws.Int64(limit)
	}

	if startID != "" {
		params.ExclusiveStartKey = map[string]*dynamodb.AttributeValue{
			"id": {S: aws.String(startID)},
		}
	}

	out, err := db.DB.Scan(params)
	if err != nil {
		return
	}

	clients = make([]osin.Client, aws.Int64Value(out.Count))

	for i, v := range out.Items {
		clients[i] = &guardian.Client{}
		err = dynamodbattribute.ConvertFromMap(v, clients[i])
		if err != nil {
			return
		}
	}

	return
}

// ListAuthorizedClients returns clients that the user has access to, based on
// the user's ldap groups and the client's admin groups. a nil user will not
// return any clients
func (db *Storage) ListAuthorizedClients(user *guardian.User, startID string, limit int64) (clients []osin.Client, err error) {
	if user == nil {
		return
	}

	clients, err = db.listAuthorizedClients(user, startID, limit)
	return
}

// ListClients returns a list of clients.
// startID can be empty, limit enables pagination, use 0 for all
func (db *Storage) ListClients(startID string, limit int64) (clients []osin.Client, err error) {
	clients, err = db.listAuthorizedClients(nil, startID, limit)
	return
}

// CompareVersion compares client and access data versions
func CompareVersion(data *osin.AccessData) (ok bool) {
	if data == nil || data.Client == nil {
		return false
	}
	return data.Client.GetVersion() == data.Version
}
