package storage

import (
	"testing"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/mocks"
	gmocks "code.justin.tv/systems/guardian/mocks"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

type storageTestObject struct {
	dynamo     *gmocks.DynamoDBAPI
	identifier *mocks.Identifier
	storage    *Storage
}

func newStorageTestObject() (sto *storageTestObject) {
	mockedDynamo := new(gmocks.DynamoDBAPI)
	mockedIdentifier := new(mocks.Identifier)
	return &storageTestObject{
		dynamo:     mockedDynamo,
		identifier: mockedIdentifier,
		storage: &Storage{
			Config:     cfg.DefaultConfig().DB,
			DB:         mockedDynamo,
			Identifier: mockedIdentifier,
		},
	}
}

func TestClientStorage(t *testing.T) {
	t.Run("DeleteClientByID", func(t *testing.T) {
		t.Run("nil user", func(t *testing.T) {
			sto := newStorageTestObject()
			err := sto.storage.DeleteClientByID(nil, "someID")
			assert.NoError(t, err)
			sto.dynamo.AssertExpectations(t)
		})

		t.Run("user with no groups", func(t *testing.T) {
			sto := newStorageTestObject()
			user := &guardian.User{}
			err := sto.storage.DeleteClientByID(user, "someID")
			assert.NoError(t, err)
			sto.dynamo.AssertExpectations(t)
		})
	})

	t.Run("GetAuthorizedClient", func(t *testing.T) {
		const clientID = "testClientID"
		user := &guardian.User{
			Groups: []string{"team-something"},
		}

		t.Run("unauthorized", func(t *testing.T) {
			sto := newStorageTestObject()
			client := &guardian.Client{
				AdminGroups: []string{"team-another-team"},
			}

			item, err := dynamodbattribute.ConvertToMap(*client)
			if err != nil {
				t.Fatal(err)
			}

			sto.dynamo.On("GetItem", mock.Anything).Return(&dynamodb.GetItemOutput{Item: item}, nil).Once()

			returnedClient, err := sto.storage.GetAuthorizedClient(user, clientID)
			assert.Error(t, err)
			assert.Nil(t, returnedClient)
			sto.dynamo.AssertExpectations(t)
		})

		t.Run("authorized", func(t *testing.T) {
			sto := newStorageTestObject()
			client := &guardian.Client{
				AdminGroups: user.Groups,
			}
			item, err := dynamodbattribute.ConvertToMap(*client)
			if err != nil {
				t.Fatal(err)
			}

			sto.dynamo.On("GetItem", mock.Anything).Return(&dynamodb.GetItemOutput{Item: item}, nil).Once()

			returnedClient, err := sto.storage.GetAuthorizedClient(user, clientID)
			assert.NoError(t, err)
			assert.NotNil(t, returnedClient)
			sto.dynamo.AssertExpectations(t)
		})
	})

	Convey("ListAuthorizedClients", t, func() {
		sto := newStorageTestObject()
		Convey("with nil user", func() {
			clients, err := sto.storage.ListAuthorizedClients(nil, "", 0)
			So(err, ShouldBeNil)
			So(clients, ShouldBeEmpty)
		})

		Convey("with user with no groups", func() {
			user := &guardian.User{
				Groups: []string{},
			}
			clients, err := sto.storage.ListAuthorizedClients(user, "", 0)
			So(err, ShouldBeNil)
			So(clients, ShouldBeEmpty)
		})
	})

	Convey("prepareAdminFilter", t, func() {
		type fixture struct {
			name                      string
			groups                    []string
			expression                string
			expressionAttributeValues map[string]*dynamodb.AttributeValue
		}

		tests := []fixture{
			{
				name:       "with some groups",
				groups:     []string{"group1", "group2", "group3"},
				expression: "contains(admin_groups, :0) or contains(admin_groups, :1) or contains(admin_groups, :2)",
				expressionAttributeValues: map[string]*dynamodb.AttributeValue{
					":0": {S: aws.String("group1")},
					":1": {S: aws.String("group2")},
					":2": {S: aws.String("group3")},
				},
			},
			{
				name:       "with one group",
				groups:     []string{"group1"},
				expression: "contains(admin_groups, :0)",
				expressionAttributeValues: map[string]*dynamodb.AttributeValue{
					":0": {S: aws.String("group1")},
				},
			},
			{
				name:       "with no groups",
				groups:     []string{},
				expression: "",
			},
		}

		for _, test := range tests {
			Convey(test.name, func() {
				filterExpression, expressionAttributeValues := prepareAdminFilter(test.groups)
				So(filterExpression, ShouldEqual, test.expression)
				So(expressionAttributeValues, ShouldResemble, test.expressionAttributeValues)
			})
		}
	})
}

func TestIntegrationClientStorage(t *testing.T) {
	storage := CreateTestDB()
	client, err := TestClient(storage)
	if err != nil {
		t.Fatal(err)
	}

	authorizedClient, err := NewTestClient()
	if err != nil {
		t.Fatal(err)
	}

	authorizedClient.AdminGroups = []string{
		"team-one",
		"team-two",
	}
	err = storage.SaveClient(authorizedClient)
	if err != nil {
		t.Fatal(err)
	}

	t.Run("save client", func(t *testing.T) {
		err := storage.SaveClient(client)
		assert.NoError(t, err)
	})

	t.Run("list authorized clients", func(t *testing.T) {
		t.Run("authorized", func(t *testing.T) {
			user := &guardian.User{
				Groups: []string{
					"team-one",
				},
			}
			clients, err := storage.ListAuthorizedClients(user, "", 0)
			assert.NoError(t, err)
			assert.Len(t, clients, 1)

			fetchedClient, ok := clients[0].(*guardian.Client)
			assert.True(t, ok)
			assert.Empty(t, fetchedClient.Secret)

			authorizedClient.Secret = ""
			assert.Equal(t, authorizedClient, fetchedClient)
		})

		t.Run("unauthorized", func(t *testing.T) {
			user := &guardian.User{
				Groups: []string{},
			}
			clients, err := storage.ListAuthorizedClients(user, "", 0)
			assert.NoError(t, err)
			assert.Empty(t, clients)
		})
	})

	t.Run("getClient", func(t *testing.T) {
		data, err := storage.GetClient(client.GetID())
		assert.NoError(t, err)

		c, ok := data.(*guardian.Client)
		assert.True(t, ok)
		assert.Empty(t, c.Secret)

		client.Secret = ""
		assert.Equal(t, client, c)
	})

	t.Run("deleteClient", func(t *testing.T) {
		t.Run("unauthorized delete", func(t *testing.T) {
			user := &guardian.User{}
			err := storage.DeleteClientByID(user, client.ID)
			assert.NoError(t, err)

			retrievedClient, err := storage.getClient(client.ID)
			assert.NoError(t, err)
			assert.NotEmpty(t, retrievedClient)
		})

		err := storage.DeleteClientByID(adminUser, authorizedClient.GetID())
		assert.NoError(t, err)

		err = storage.DeleteClientByID(adminUser, client.GetID())
		assert.NoError(t, err)

		data, err := storage.GetClient(client.GetID())
		assert.NoError(t, err)
		assert.Nil(t, data)
	})
}
