package storage

import (
	"time"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/osin"
	awssession "github.com/aws/aws-sdk-go/aws/session"
	"github.com/satori/go.uuid"
)

var adminUser = &guardian.User{
	Groups: []string{guardian.TeamSSELDAPGroup},
}

// CreateTestDB returns a test db
func CreateTestDB() (storage *Storage) {
	c := cfg.DefaultConfig()
	c.DB.ClientsTable = cfg.DefaultTestClientsTable
	c.DB.AuthorizationsTable = cfg.DefaultTestAuthorizationsTable
	c.DB.AWSSession = awssession.New(c.DB.AWSConfig)
	storage = NewDynamoDBStorage(c, &guardian.TestIdentifier{})
	return
}

// CreateTestAuthorizeData generates a new auth code
func CreateTestAuthorizeData(client *guardian.Client) (authData *osin.AuthorizeData, err error) {
	authData = &osin.AuthorizeData{
		Client:      client,
		CreatedAt:   time.Now(),
		ExpiresIn:   0,
		UserData:    guardian.TestUser,
		State:       "state",
		RedirectURI: cfg.DefaultDevBaseURL,
		Scope:       "",
	}
	authData.Code, err = DefaultTokenGenerator.GenerateAuthorizeToken(authData)
	return
}

// CreateTestAccessData generates a new osin.AccessData. Does not persist data.
func CreateTestAccessData(client *guardian.Client) (accessData *osin.AccessData, err error) {
	accessData = &osin.AccessData{
		Client:      client,
		RedirectURI: cfg.DefaultDevBaseURL,
		CreatedAt:   time.Now(),
		ExpiresIn:   0,
		UserData:    guardian.TestUser,
	}
	accessData.AccessToken, _, err = DefaultTokenGenerator.GenerateAccessToken(accessData, false)
	return
}

// TestClient persists a new test client, and returns the new object
func TestClient(db guardian.Storer) (client *guardian.Client, err error) {
	client, err = NewTestClient()
	if err != nil {
		return
	}
	err = db.SaveClient(client)
	return
}

// NewTestClient creates a new test client, does not persist data
func NewTestClient() (client *guardian.Client, err error) {
	uuid := uuid.NewV4()
	name := "testapp" + uuid.String()
	description := "really cool test app"
	homepage := "http://twitch.tv"
	redirect := "http://localhost:8000"

	return guardian.NewClient(name, redirect, description, homepage)
}
