package asiimov

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"

	"golang.org/x/oauth2"

	"testing"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/systems/guardian/guardian/tokens"
	"github.com/jixwanwang/apiutils"
	"github.com/stretchr/testify/assert"
)

var noopLogger = log.New(ioutil.Discard, "", 0)

type mwTestObject struct {
	tokenStore map[string]*guardian.User
	token      *oauth2.Token
	cfg        *oauth2.Config
	mw         *Asiimov
	server     *httptest.Server
}

func handleCheckToken(tokenStore map[string]*guardian.User) (handler http.Handler) {
	handler = http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		token := GetToken(r)
		if token == "" {
			err := json.NewEncoder(w).Encode([]byte(`{
 "errors": [
  {
   "title": "Bad Request",
   "detail": "token required via basic auth or query param",
   "status": "400",
   "source": {
    "pointer": ""
   }
  }
 ],
 "jsonapi": {
  "version": "1.1"
 }
}`))
			if err != nil {
				// /shrug
			}
		}
		user, ok := tokenStore[token]
		if !ok {
			err := json.NewEncoder(w).Encode([]byte(`{
 "errors": [
  {
   "title": "Not Found",
   "detail": "token not found",
   "status": "404",
   "source": {
    "pointer": ""
   }
  }
 ],
 "jsonapi": {
  "version": "1.1"
 }
}`))
			if err != nil {
				// /shrug
			}
		}
		apiutils.ServeJSON(w, &guardian.TokenCheck{
			Token: &oauth2.Token{
				TokenType:   "Bearer",
				AccessToken: token,
			},
			User: user,
		})
	})
	return
}

func setupMWTestObject(t *testing.T) (mto *mwTestObject) {
	secret, _, err := tokens.DefaultTokenGenerator.GenerateSecret()
	if err != nil {
		t.Fatal(err)
	}

	mto = &mwTestObject{
		tokenStore: make(map[string]*guardian.User),
		token: &oauth2.Token{
			AccessToken: secret,
		},
		mw: &Asiimov{
			logger:           noopLogger,
			allowedGroups:    make(map[string]struct{}),
			disallowedGroups: make(map[string]struct{}),
		},
	}

	mto.server = httptest.NewServer(handleCheckToken(mto.tokenStore))
	cfg := new(oauth2.Config)
	cfg.Endpoint.AuthURL = mto.server.URL
	cfg.Endpoint.TokenURL = mto.server.URL
	mto.cfg = cfg
	mto.mw.CheckTokenURL = mto.server.URL
	mto.mw.VerifyAccess = mto.mw.DefaultVerifyAccess
	return
}

func (mto *mwTestObject) teardown() {
	mto.server.Close()
}

func TestGetToken(t *testing.T) {
	mto := setupMWTestObject(t)
	defer mto.teardown()

	t.Run("token as bearer token in Authorization header", func(t *testing.T) {
		testReq := httptest.NewRequest(http.MethodGet, mto.server.URL, nil)
		mto.token.SetAuthHeader(testReq)

		token := GetToken(testReq)
		assert.Equal(t, mto.token.AccessToken, token)
	})

	t.Run("token as password in basic auth", func(t *testing.T) {
		testReq := httptest.NewRequest(http.MethodGet, mto.server.URL, nil)
		testReq.SetBasicAuth("", mto.token.AccessToken)

		token := GetToken(testReq)
		assert.Equal(t, mto.token.AccessToken, token)
	})

	t.Run("missing token", func(t *testing.T) {
		testReq := httptest.NewRequest(http.MethodGet, mto.server.URL, nil)
		assert.Empty(t, GetToken(testReq))
	})
}

func TestDefaultVerifyAccess(t *testing.T) {
	t.Run("return error if a user in a disallowed group", func(t *testing.T) {
		mto := setupMWTestObject(t)
		defer mto.teardown()

		testGroups := []string{"team-test"}
		err := mto.mw.DisallowGroups(testGroups)
		assert.NoError(t, err)

		testUser := &guardian.User{Groups: testGroups}
		err = mto.mw.VerifyAccess(testUser)
		assert.Error(t, err)
	})

	t.Run("return error if a user is not in an allowed group", func(t *testing.T) {
		mto := setupMWTestObject(t)
		defer mto.teardown()

		testGroups := []string{"team-test"}
		err := mto.mw.AllowGroups(testGroups)
		assert.NoError(t, err)

		testUser := &guardian.User{Groups: testGroups}
		err = mto.mw.VerifyAccess(testUser)
		assert.NoError(t, err)
	})

	t.Run("return error if a user is in both allowed and disallowed groups", func(t *testing.T) {
		mto := setupMWTestObject(t)
		defer mto.teardown()

		const (
			allowedGroup    = "team-allowed"
			disallowedGroup = "team-disallowed"
		)
		testUser := &guardian.User{
			Groups: []string{allowedGroup, disallowedGroup},
		}
		err := mto.mw.AllowGroups([]string{allowedGroup})
		assert.NoError(t, err)
		err = mto.mw.DisallowGroups([]string{disallowedGroup})
		assert.NoError(t, err)

		err = mto.mw.VerifyAccess(testUser)
		assert.Error(t, err)
	})

	t.Run("success", func(t *testing.T) {
		mto := setupMWTestObject(t)
		defer mto.teardown()

		testGroups := []string{"team-allowed"}
		testUser := &guardian.User{
			Groups: testGroups,
		}
		err := mto.mw.AllowGroups(testGroups)
		assert.NoError(t, err)
		err = mto.mw.VerifyAccess(testUser)
		assert.NoError(t, err)
	})
}

func TestHandleGuardianRequest(t *testing.T) {
	t.Run("success", func(t *testing.T) {
		mto := setupMWTestObject(t)
		defer mto.teardown()

		testUser := &guardian.User{
			UID:    "guardian",
			Groups: []string{"team-test"},
		}
		mto.tokenStore[mto.token.AccessToken] = testUser

		req := httptest.NewRequest(http.MethodGet, mto.server.URL, nil)
		mto.token.SetAuthHeader(req)

		user, err := mto.mw.HandleGuardianRequest(req)
		assert.NoError(t, err)

		assert.Equal(t, testUser, user)
	})
}
