package nonce

import (
	"sync"
	"time"

	"github.com/satori/go.uuid"
)

// DefaultMap is a Map with a 30 second nonce validity period
var DefaultMap = NewMap(30 * time.Second)

// NewMap initializes a Store backed by a map
func NewMap(timeout time.Duration) *Map {
	return &Map{
		cache:   make(map[string]time.Time),
		Timeout: timeout,
	}
}

// Map implements Store with an in memory map and mutex
// lock.
type Map struct {
	sync.RWMutex
	cache   map[string]time.Time
	Timeout time.Duration
}

// Generate generates a UUID and commits it to the map.
func (t *Map) Generate() (nonce string, err error) {
	nonce = uuid.NewV4().String()
	t.Lock()
	defer t.Unlock()

	if t.cache == nil {
		t.cache = make(map[string]time.Time)
	}

	t.cache[nonce] = time.Now().UTC()
	return
}

// Check checks to see if the nonce is in cache and within timeout
// (5 minutes)
func (t *Map) Check(nonce string) (valid bool, err error) {
	t.RLock()
	defer t.RUnlock()

	if t.Timeout == 0 {
		t.Timeout = time.Minute * 5
	}

	var timestamp time.Time
	if timestamp, valid = t.cache[nonce]; !valid {
		return
	}

	duration := time.Now().UTC().Sub(timestamp)
	if duration > t.Timeout {
		valid = false
		return
	}

	valid = true
	delete(t.cache, nonce)
	return
}
