package nonce

import (
	"time"

	"github.com/garyburd/redigo/redis"
	"github.com/satori/go.uuid"
)

// Redis implements Store backed by redis.
type Redis struct {
	Pool         *redis.Pool
	NonceTimeout time.Duration
}

// NewRedisPool initializes a pool of redis connections
func NewRedisPool(redisURL string) (pool *redis.Pool, err error) {
	pool = redis.NewPool(func() (c redis.Conn, err error) {
		c, err = redis.DialURL(redisURL)
		if err != nil {
			c.Close()
			return
		}
		return
	}, 3)
	return
}

// Generate generates a UUID and commits it to redis.
func (rn *Redis) Generate() (nonce string, err error) {
	c := rn.Pool.Get()
	defer func() {
		closeErr := c.Close()
		if closeErr != nil {
			err = closeErr
		}
	}()

	nonce = uuid.NewV4().String()
	_, err = c.Do("SETEX", nonce, rn.NonceTimeout.Seconds(), 0)
	if err != nil {
		return
	}
	return
}

// Check ensures that the nonce exists in redis and has not been used and has
// not expired
func (rn *Redis) Check(nonce string) (valid bool, err error) {
	c := rn.Pool.Get()
	defer func() {
		closeErr := c.Close()
		if closeErr != nil {
			err = closeErr
		}
	}()

	deleted, err := redis.Int64(c.Do("DEL", nonce))
	if err != nil {
		return
	}

	if deleted > 0 {
		valid = true
	}
	return
}
