// +build integration

package agent

import (
	"encoding/json"
	"testing"

	managerMocks "code.justin.tv/systems/sandstorm/manager/mocks"
	globalMocks "code.justin.tv/systems/sandstorm/mocks"
	"code.justin.tv/systems/sandstorm/queue"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

type mockedQueueMessage struct {
	RawSecret string `json:"message"`
}

func getMockedAgent() (*Agent, *globalMocks.SQSAPI, *managerMocks.API) {
	testFolderPath := prepareTestPath("agent-basic-test")
	agent := getTestAgent(testFolderPath, true, false)

	mockedSQS := new(globalMocks.SQSAPI)

	mockedManager := new(managerMocks.API)

	agent.Queue.SQS = mockedSQS
	agent.manager = mockedManager

	return agent, mockedSQS, mockedManager
}

func TestPollForSecrets(t *testing.T) {

	agent, mockedSQS, _ := getMockedAgent()
	secretName := "systems/sandstorm-agent/development/poll_test_123"
	t.Run("agent.pollForSecret tests", func(t *testing.T) {
		t.Run("Poll for secret", func(t *testing.T) {

			secret := &queue.Secret{}
			secret.UpdatedAt.S = "123456"
			secret.Name.S = secretName

			rawSecret, err := json.Marshal(secret)
			assert.Nil(t, err, "Failed to marshal queue secret")

			queueMessage := &queue.Message{
				RawSecret: string(rawSecret[:]),
			}

			queueMessageBody, err := json.Marshal(queueMessage)
			assert.Nil(t, err, "Failed to marshal queue message")

			messageBody := string(queueMessageBody[:])
			sqsHandler := "handler"
			outPutMessage := &sqs.ReceiveMessageOutput{
				Messages: []*sqs.Message{
					&sqs.Message{
						Body:          &messageBody,
						ReceiptHandle: &sqsHandler,
					},
				},
			}

			// So that the agent stop polling
			noOutputMessage := &sqs.ReceiveMessageOutput{
				Messages: []*sqs.Message{},
			}
			whitelist := []string{
				secretName,
			}

			mockedSQS.On("ReceiveMessageWithContext", mock.Anything, mock.Anything, mock.Anything).Return(outPutMessage, nil).Once()

			mockedSQS.On("ReceiveMessageWithContext", mock.Anything, mock.Anything, mock.Anything).Return(noOutputMessage, nil).Once()
			mockedSQS.On("DeleteMessage", mock.Anything).Return(nil, nil).Times(2)

			secrets := agent.pollForSecrets(whitelist)
			assert.Equal(t, 1, len(secrets), "Expected to recieve an update for 1 secret.")

		})
	})
}
