package agent

import (
	"path"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestStateFile(t *testing.T) {

	Convey("Test StateFile", t, func() {

		testFolderPath := prepareTestPath("test-statefile")

		Convey("->addTemplates()", func() {

			stateFile := &StateFile{}
			stateFile.Templates = map[string]*Template{}

			Convey("should properly copy templates and secrets", func() {

				testTemplate := getTestTemplate(testFolderPath)

				// to make sure the test will actually work
				for _, secret := range testTemplate.Secrets {
					So(secret.Plaintext, ShouldNotBeEmpty)
				}

				stateFile.AddTemplates([]*Template{testTemplate})
				So(len(stateFile.Templates), ShouldEqual, 1)

				// state file plaintext should be blank because we don't want to store
				// secrets to file, agent secret plaintext shouldn't get KO'd in the
				// process
				for _, stateTemplate := range stateFile.Templates {
					for _, secret := range stateTemplate.Secrets {
						So(secret.Plaintext, ShouldBeEmpty)
					}
				}

				for _, secret := range testTemplate.Secrets {
					So(secret.Plaintext, ShouldNotBeEmpty)
				}
			})
		})

		Convey("->newStateFile()", func() {
			stateFile := newStateFile([]*Template{getTestTemplate(testFolderPath)})
			So(stateFile.LastUpdate, ShouldEqual, 0)
			So(len(stateFile.Templates), ShouldEqual, 1)
		})

		Convey("->ToFile()", func() {

			stateFilePath := path.Join(testFolderPath, "state-tofile-test")

			Convey("should output successfully", func() {
				stateFile := newStateFile([]*Template{getTestTemplate(testFolderPath)})

				err := stateFile.ToFile(stateFilePath)
				So(err, ShouldBeNil)
				So(fileExists(stateFilePath), ShouldBeTrue)
				So(stateFile.LastUpdate, ShouldNotEqual, 0)
			})
		})

		Convey("->parseStateFile()", func() {

			testTemplate := getTestTemplate(testFolderPath)
			stateFilePath := path.Join(testFolderPath, "state-parse-test")
			stateFile := newStateFile([]*Template{testTemplate})

			err := stateFile.ToFile(stateFilePath)
			So(err, ShouldBeNil)

			Convey("should parse successfully", func() {

				stateFile, err := parseStateFile(stateFilePath)
				So(err, ShouldBeNil)

				So(len(stateFile.Templates), ShouldEqual, 1)
				So(stateFile.LastUpdate, ShouldBeGreaterThan, 0)

				for _, template := range stateFile.Templates {

					So(template.Source, ShouldNotBeNil)
					So(template.Destination, ShouldNotBeNil)
					So(template.Command, ShouldNotBeNil)
					So(template.CommandStatus.Status, ShouldEqual, statusStartup)
					So(template.Secrets, ShouldNotBeEmpty)
					So(template.Content, ShouldNotBeNil)

					for name, secret := range template.Secrets {
						_, exists := testTemplate.Secrets[name]
						So(exists, ShouldBeTrue)
						So(secret.UpdatedAt, ShouldNotBeNil)
					}
				}
			})
		})
	})
}
