// +build integration

package apiserver

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"path/filepath"
	"strings"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"code.justin.tv/systems/sandstorm/policy"
	"code.justin.tv/systems/sandstorm/testutil"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/iam"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
	"golang.org/x/oauth2"
)

const (
	jchenArn = "arn:aws:iam::734326455073:user/jchen"
	bachArn  = "arn:aws:iam::734326455073:user/bach"

	teamNavi = "team-navi"

	// BEWARE: testSecret1 and testSecret2 are used in PUT testing as secrets namespaced under syseng.
	testSecret1         = "syseng/testerino/testing/test_secret"
	testWildcardSecret  = "syseng/testerino/testing/*"
	testWildcardSecret2 = "syseng/testerino/testing/blah*"
	testSecret2         = "syseng/testerino/testing/test_secret_2"
	devSecret           = "syseng/testerino/dev/dev_secret"

	disallowedTestSecret  = "nonExistentTeam/testerino/testing/test_secret"
	testRWAuxPolicyArn    = "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-rw-aux"
	testAuxPolicyArn      = "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-aux"
	testNamespaceIndexArn = "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing/index/namespace_name"
	testNamespaceTableArn = "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_namespaces"
	testSecretsAuditTable = "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_audit"
	testSecretsTable      = "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing"
)

var invalidWildcardSecrets = []string{
	"syseng/*/testing/*",
	"*/app/env/*",
}

var testRoleID = fmt.Sprintf("sandstorm-api-test-%s", uuid.NewV4().String())

func getPolicyAttachedToRole(roleName string) (*iam.ListAttachedRolePoliciesOutput, error) {
	awsConfig := getAwsCredentialConfig()

	sess := session.New(awsConfig)
	iamInstance := iam.New(sess)

	params := &iam.ListAttachedRolePoliciesInput{
		RoleName: aws.String(roleName),
	}

	resp, err := iamInstance.ListAttachedRolePolicies(params)
	if err != nil {
		return nil, err
	}
	return resp, nil
}

func getCreateRoleRequest(arn string, roleName string, writeAccess bool, owners []string) *policy.CreateRoleRequest {
	role := &policy.CreateRoleRequest{
		AllowedArns: []string{arn},
		Name:        roleName,
		SecretKeys: []string{
			testSecret1,
			testWildcardSecret,
			testWildcardSecret2,
		},
		WriteAccess: writeAccess,
		Owners:      owners,
	}
	return role
}

func TestPolicyAPI(t *testing.T) {
	Convey("API Tests", t, func() {
		m := createTestManager()

		tmpdir, err := ioutil.TempDir("", "changelogClient")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(tmpdir)
			if err != nil {
				t.Error(err)
			}
		}()

		logDir, err := ioutil.TempDir("", "logDir")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(logDir)
			if err != nil {
				t.Error(err)
			}
		}()

		cfg, err := LoadConfig("../test.hcl")
		if err != nil {
			t.Fatal(err)
		}
		cfg.Changelog.BoltDBFilePath = filepath.Join(tmpdir, "test.boltdb")
		cfg.Changelog.LogDir = logDir

		svc, err := New(cfg)
		if err != nil {
			t.Fatal(err)
		}
		server := httptest.NewServer(svc)
		defer server.Close()

		baseURL := server.URL
		mgr := svc.clients.mgr.(*manager.Manager)

		Convey("KMS key should be available", func() {
			err := mgr.CheckKMS()
			So(err, ShouldBeNil)
		})
		Convey("DynamoDB tables should be accessible", func() {
			err := mgr.CheckTable()
			So(err, ShouldBeNil)
		})

		Convey("Policy Generator Tests", func() {
			role := &policy.CreateRoleRequest{
				AllowedArns: []string{jchenArn},
				Name:        testRoleID,
				SecretKeys: []string{
					testSecret1,
					testWildcardSecret,
					testWildcardSecret2,
				},
				WriteAccess: true,
				Owners:      []string{teamNavi},
			}
			bs, encodeErr := json.Marshal(role)
			So(encodeErr, ShouldBeNil)
			owners := policy.NewOwners(role.Owners)
			var err error

			Convey("POST /roles", func() {
				Convey("is invalid when", func() {
					Convey("owners field isn't in request", func() {
						badRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Name:        testRoleID,
							SecretKeys:  []string{testSecret1},
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, 422)
					})
					Convey(" name field isn't in request", func() {
						badRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							SecretKeys:  []string{testSecret1},
							Owners:      []string{teamNavi},
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, 422)
					})
					Convey(" AllowedARNs field isn't in request", func() {
						badRole := &policy.CreateRoleRequest{
							SecretKeys: []string{testSecret1},
							Owners:     []string{teamNavi},
							Name:       testRoleID,
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, 422)
					})
					Convey(" SecretKeys field isn't in request", func() {
						badRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Owners:      []string{teamNavi},
							Name:        testRoleID,
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, 422)
					})

					Convey("a wildcard is in an invalid field", func() {
						badRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Owners:      []string{teamNavi},
							Name:        testRoleID,
							SecretKeys:  invalidWildcardSecrets,
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusForbidden)

						jsonError := JSONErrors{}
						So(json.NewDecoder(resp.Body).Decode(&jsonError), ShouldBeNil)
						So(jsonError.Errors[0].Class, ShouldEqual, "ErrDisallowedWildcard")
					})

					Convey("multiple environments are sent", func() {
						badRole := &policy.CreateRoleRequest{
							SecretKeys: []string{
								"myGroup/myProject/production/secretName",
								"myGroup/myProject/development/secretName",
							},
							AllowedArns: []string{jchenArn},
							Owners:      []string{teamNavi},
							Name:        testRoleID,
						}
						bs, encodeErr := json.Marshal(badRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, 403)
					})
				})

				Convey("with a valid request should create roles and policies", func() {
					resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusCreated)

					createRoleResponse := new(policy.CreateRoleResponse)
					decodeErr := json.NewDecoder(resp.Body).Decode(createRoleResponse)
					So(decodeErr, ShouldBeNil)
					So(resp.Body.Close(), ShouldBeNil)
					So(createRoleResponse.RoleArn, ShouldNotBeEmpty)

					expectedResponse := &policy.CreateRoleResponse{
						RoleArn:  createRoleResponse.RoleArn,
						RoleName: testRoleID,
						Policy: policy.IAMPolicyDocument{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMPolicyStatement{
								policy.IAMPolicyStatement{
									Action:   policy.RWNamespaceIndexActions,
									Resource: []interface{}{testNamespaceIndexArn},
									Effect:   "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringEquals": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												"syseng",
											},
										},
									},
									Sid: policy.NamespaceIndexSid,
								},
								policy.IAMPolicyStatement{
									Action: policy.RWSecretsTableActions,
									Resource: []interface{}{
										testSecretsTable,
										testSecretsAuditTable,
									},
									Effect: "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringLike": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												testSecret1,
												testWildcardSecret,
												testWildcardSecret2,
											},
										},
									},
									Sid: policy.SecretsTableSid,
								},
								policy.IAMPolicyStatement{
									Action:   []interface{}{"sts:AssumeRole"},
									Resource: []interface{}{createRoleResponse.RoleArn},
									Effect:   "Allow",
								},
								policy.IAMPolicyStatement{
									Action:   policy.RWNamespaceTableActions,
									Resource: []interface{}{testNamespaceTableArn},
									Effect:   "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringEquals": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												"syseng",
											},
										},
									},
									Sid: policy.NamespaceTableSid,
								},
							},
						},
						AssumeRolePolicy: policy.IAMAssumeRolePolicy{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMAssumeRoleStatement{
								policy.IAMAssumeRoleStatement{
									Effect: "Allow",
									Action: "sts:AssumeRole",
									Resource: []interface{}{
										createRoleResponse.RoleArn,
									},
								},
							},
						},
						AllowedArns:  role.AllowedArns,
						AuxPolicyArn: testRWAuxPolicyArn,
						WriteAccess:  true,
						Owners:       owners,
					}
					So(createRoleResponse, ShouldResemble, expectedResponse)
				})

				Convey("multiple envs should be ok if cross-env is set", func() {
					var crossEnvRoleName = fmt.Sprintf("sandstorm-api-test-%s", uuid.NewV4().String())
					crossEnvSecretName := testutil.GetRandomSecretNameWithPrefix(
						"syseng/testerino/testing/dev",
					)

					var crossEnvRoleID string

					defer func() {
						_, err := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", crossEnvRoleID), "DELETE", nil, true)
						So(err, ShouldBeNil)
					}()

					err := m.Post(&manager.Secret{
						Name:      crossEnvSecretName,
						Plaintext: []byte("blahh"),
						CrossEnv:  true,
					})
					So(err, ShouldBeNil)

					getExpectedResponse := func(roleArn string) *policy.CreateRoleResponse {
						return &policy.CreateRoleResponse{
							RoleArn:  roleArn,
							RoleName: crossEnvRoleName,
							Policy: policy.IAMPolicyDocument{
								Version: policy.DocumentVersion,
								Statement: []policy.IAMPolicyStatement{
									policy.IAMPolicyStatement{
										Action:   policy.RWNamespaceIndexActions,
										Resource: []interface{}{testNamespaceIndexArn},
										Effect:   "Allow",
										Condition: policy.IAMStatementCondition{
											"ForAllValues:StringEquals": map[string]interface{}{
												"dynamodb:LeadingKeys": []interface{}{
													"syseng",
												},
											},
										},
										Sid: policy.NamespaceIndexSid,
									},
									policy.IAMPolicyStatement{
										Action: policy.RWSecretsTableActions,
										Resource: []interface{}{
											testSecretsTable,
											testSecretsAuditTable,
										},
										Effect: "Allow",
										Condition: policy.IAMStatementCondition{
											"ForAllValues:StringLike": map[string]interface{}{
												"dynamodb:LeadingKeys": []interface{}{
													testSecret1,
													testWildcardSecret,
													testWildcardSecret2,
													crossEnvSecretName,
												},
											},
										},
										Sid: policy.SecretsTableSid,
									},
									policy.IAMPolicyStatement{
										Action:   []interface{}{"sts:AssumeRole"},
										Resource: []interface{}{roleArn},
										Effect:   "Allow",
									},
									policy.IAMPolicyStatement{
										Action:   policy.RWNamespaceTableActions,
										Resource: []interface{}{testNamespaceTableArn},
										Effect:   "Allow",
										Condition: policy.IAMStatementCondition{
											"ForAllValues:StringEquals": map[string]interface{}{
												"dynamodb:LeadingKeys": []interface{}{
													"syseng",
												},
											},
										},
										Sid: policy.NamespaceTableSid,
									},
								},
							},
							AssumeRolePolicy: policy.IAMAssumeRolePolicy{
								Version: policy.DocumentVersion,
								Statement: []policy.IAMAssumeRoleStatement{
									policy.IAMAssumeRoleStatement{
										Effect: "Allow",
										Action: "sts:AssumeRole",
										Resource: []interface{}{
											roleArn,
										},
									},
								},
							},
							AllowedArns:  role.AllowedArns,
							AuxPolicyArn: testRWAuxPolicyArn,
							WriteAccess:  true,
							Owners:       owners,
						}
					}

					Convey("when posted", func() {
						role := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Name:        crossEnvRoleName,
							SecretKeys: []string{
								testSecret1,
								testWildcardSecret,
								testWildcardSecret2,
								crossEnvSecretName,
							},
							WriteAccess: true,
							Owners:      []string{teamNavi},
						}
						bs, encodeErr := json.Marshal(role)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusCreated)
						createRoleResponse := new(policy.CreateRoleResponse)
						decodeErr := json.NewDecoder(resp.Body).Decode(createRoleResponse)
						So(decodeErr, ShouldBeNil)
						So(resp.Body.Close(), ShouldBeNil)
						So(createRoleResponse.RoleArn, ShouldNotBeEmpty)

						crossEnvRoleID = createRoleResponse.RoleName

						expectedResponse := getExpectedResponse(createRoleResponse.RoleArn)
						assert.Equal(t, expectedResponse, createRoleResponse)

						Convey("when updated", func() {
							putRoleRequest := &policy.PutRoleRequest{
								AllowedArns: []string{jchenArn},
								SecretKeys: []string{
									testSecret1,
									testWildcardSecret,
									testWildcardSecret2,
									crossEnvSecretName,
								},
								WriteAccess: true,
								Owners:      []string{teamNavi},
							}
							bs, encodeErr := json.Marshal(putRoleRequest)
							So(encodeErr, ShouldBeNil)

							roleURI := fmt.Sprintf("/roles/%s", crossEnvRoleID)
							resp, respErr := testJSONRequest(baseURL, roleURI, "POST", bs, true)
							So(respErr, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusOK)

							putResponse := new(policy.PutRoleResponse)
							decodeErr := json.NewDecoder(resp.Body).Decode(putResponse)
							So(decodeErr, ShouldBeNil)
							So(resp.Body.Close(), ShouldBeNil)

							So(putResponse.RoleArn, ShouldEqual, createRoleResponse.RoleArn)
							So(crossEnvRoleName, ShouldEqual, createRoleResponse.RoleName)
							So(putResponse.Policy, ShouldResemble, createRoleResponse.Policy)
							So(putResponse.AssumeRolePolicy, ShouldResemble, createRoleResponse.AssumeRolePolicy)
							So(putResponse.AllowedArns, ShouldResemble, createRoleResponse.AllowedArns)
							So(putResponse.WriteAccess, ShouldEqual, createRoleResponse.WriteAccess)
						})
					})
				})
			})

			Convey("GET /roles", func() {
				resp, respErr := testJSONRequest(baseURL, "/roles", "GET", nil, true)
				So(respErr, ShouldBeNil)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)

				var listRoleResponse []string
				decodeErr := json.NewDecoder(resp.Body).Decode(&listRoleResponse)
				So(decodeErr, ShouldBeNil)
				So(resp.Body.Close(), ShouldBeNil)
				So(len(listRoleResponse), ShouldBeGreaterThan, 0)
			})

			Convey("GET /roles/{role}", func() {
				Convey("GET /roles/{existing-role} should return the role", func() {
					resp, respErr := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", role.Name), "GET", nil, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					getRoleResponse := new(policy.IAMRole)
					decodeErr := json.NewDecoder(resp.Body).Decode(getRoleResponse)
					So(decodeErr, ShouldBeNil)
					So(resp.Body.Close(), ShouldBeNil)
					So(getRoleResponse, ShouldNotBeEmpty)
					So(getRoleResponse.RoleArn, ShouldNotBeEmpty)
					expectedResponse := &policy.IAMRole{
						RoleArn:  getRoleResponse.RoleArn,
						RoleName: testRoleID,
						Policy: policy.IAMPolicyDocument{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMPolicyStatement{
								policy.IAMPolicyStatement{
									Action:   policy.RWNamespaceIndexActions,
									Resource: []interface{}{testNamespaceIndexArn},
									Effect:   "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringEquals": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												"syseng",
											},
										},
									},
									Sid: policy.NamespaceIndexSid,
								},
								policy.IAMPolicyStatement{
									Action: policy.RWSecretsTableActions,
									Resource: []interface{}{
										testSecretsTable,
										testSecretsAuditTable,
									},
									Effect: "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringLike": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												testSecret1,
												testWildcardSecret,
												testWildcardSecret2,
											},
										},
									},
									Sid: policy.SecretsTableSid,
								},
								policy.IAMPolicyStatement{
									Action:   []interface{}{"sts:AssumeRole"},
									Resource: []interface{}{getRoleResponse.RoleArn},
									Effect:   "Allow",
								},
								policy.IAMPolicyStatement{
									Action:   policy.RWNamespaceTableActions,
									Resource: []interface{}{testNamespaceTableArn},
									Effect:   "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringEquals": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												"syseng",
											},
										},
									},
									Sid: policy.NamespaceTableSid,
								},
							},
						},
						AssumeRolePolicy: policy.IAMAssumeRolePolicy{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMAssumeRoleStatement{
								policy.IAMAssumeRoleStatement{
									Effect: "Allow",
									Action: "sts:AssumeRole",
									Resource: []interface{}{
										getRoleResponse.RoleArn,
									},
								},
							},
						},
						AllowedArns:  role.AllowedArns,
						AuxPolicyArn: testRWAuxPolicyArn,
						WriteAccess:  true,
						Owners:       owners,
						Group:        "",
					}
					So(getRoleResponse, ShouldResemble, expectedResponse)
				})

				Convey("GET /roles/{nonexistent-role} should return a 404", func() {
					resp, respErr := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", uuid.NewV4().String()), "GET", nil, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
				})
			})

			Convey("POST /roles/{role}", func() {
				Convey("is invalid when", func() {
					Convey("role doesn't exist", func() {
						putRoleRequest := &policy.PutRoleRequest{
							SecretKeys:  []string{testSecret2},
							AllowedArns: []string{bachArn},
						}
						bs, encodeErr := json.Marshal(putRoleRequest)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", uuid.NewV4().String()), "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusNotFound)

					})
					Convey("adding owner when a roleOwner doesn't exist, and not admin", func() {

						// Create a role
						tempRoleName := uuid.NewV4().String()
						tempRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Name:        tempRoleName,
							SecretKeys: []string{
								testSecret1,
								testWildcardSecret,
								testWildcardSecret2,
							},
							WriteAccess: true,
							Owners:      []string{teamNavi},
						}
						bs, encodeErr := json.Marshal(tempRole)
						So(encodeErr, ShouldBeNil)
						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusCreated)

						// Delete roleOwner
						pg := &policy.IAMPolicyGenerator{
							AuxPolicyArn:                 "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-aux",
							RWAuxPolicyArn:               "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-rw-aux",
							DynamoDBSecretsTableArn:      "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing",
							DynamoDBSecretsAuditTableArn: "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_audit",
							DynamoDBNamespaceTableArn:    "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_namespaces",
							RoleOwnerTableName:           "sandstorm-testing_role_owners",
						}
						pg.SetPathPrefixEnvironment("testing")
						awsConfig := &aws.Config{Region: aws.String("us-west-2")}
						sess := session.New(awsConfig)
						stsclient := sts.New(sess)
						arp := &stscreds.AssumeRoleProvider{
							Duration:     900 * time.Second,
							ExpiryWindow: 10 * time.Second,
							RoleARN:      "arn:aws:iam::734326455073:role/sandstorm-apiserver-testing",
							Client:       stsclient,
						}
						credentials := credentials.NewCredentials(arp)
						awsConfig.WithCredentials(credentials)

						sess = session.New(awsConfig)
						pg.IAM = iam.New(sess)
						pg.DynamoDB = dynamodb.New(sess)

						err = pg.DeleteRoleOwners(tempRoleName, nil)
						So(err, ShouldBeNil)
						// try to add owner as non-admin
						tempRolePut := &policy.PutRoleRequest{
							SecretKeys:  []string{testSecret2},
							AllowedArns: []string{bachArn},
							Owners:      []string{teamNavi},
							WriteAccess: false,
						}

						bs, encodeErr = json.Marshal(tempRolePut)
						So(encodeErr, ShouldBeNil)

						assembledURL := strings.Join([]string{baseURL, fmt.Sprintf("/roles/%s", tempRoleName)}, "")
						req, err := http.NewRequest("POST", assembledURL, bytes.NewBuffer(bs))
						So(err, ShouldBeNil)

						req.Header.Set("Origin", "http://localhost")
						req.Header.Set("Content-Type", contentType)
						// Setting access token for user that doesn't have access to syseng
						token := &oauth2.Token{AccessToken: LimitedAccessToken}
						token.SetAuthHeader(req)

						client := &http.Client{}
						failedResp, failerr := client.Do(req)
						So(failerr, ShouldBeNil)
						// resp, respErr = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRole.Name), "POST", bs, true)
						So(failedResp.StatusCode, ShouldEqual, http.StatusForbidden)

						// cleanup
						resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRole.Name), "DELETE", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
					})
				})

				Convey("with a valid role should update the role", func() {
					putRoleRequest := &policy.PutRoleRequest{
						SecretKeys:  []string{testSecret2},
						AllowedArns: []string{bachArn},
						Owners:      []string{teamNavi},
						WriteAccess: false,
					}
					bs, encodeErr := json.Marshal(putRoleRequest)
					So(encodeErr, ShouldBeNil)

					resp, respErr := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", role.Name), "POST", bs, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					putRoleResponse := new(policy.PutRoleResponse)
					decodeErr := json.NewDecoder(resp.Body).Decode(putRoleResponse)
					So(decodeErr, ShouldBeNil)
					So(resp.Body.Close(), ShouldBeNil)
					So(putRoleResponse.RoleArn, ShouldNotBeEmpty)
					expectedResponse := &policy.PutRoleResponse{
						RoleArn:  putRoleResponse.RoleArn,
						RoleName: testRoleID,
						Policy: policy.IAMPolicyDocument{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMPolicyStatement{
								policy.IAMPolicyStatement{
									Action:   []interface{}{"dynamodb:Query"},
									Resource: []interface{}{testNamespaceIndexArn},
									Effect:   "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringEquals": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												"syseng",
											},
										},
									},
									Sid: policy.NamespaceIndexSid,
								},
								policy.IAMPolicyStatement{
									Action: []interface{}{
										"dynamodb:Query",
										"dynamodb:DescribeTable",
										"dynamodb:GetItem",
									},
									Resource: []interface{}{
										testSecretsTable,
									},
									Effect: "Allow",
									Condition: policy.IAMStatementCondition{
										"ForAllValues:StringLike": map[string]interface{}{
											"dynamodb:LeadingKeys": []interface{}{
												testSecret2,
											},
										},
									},
									Sid: policy.SecretsTableSid,
								},
								policy.IAMPolicyStatement{
									Action:   []interface{}{"sts:AssumeRole"},
									Resource: []interface{}{putRoleResponse.RoleArn},
									Effect:   "Allow",
								},
							},
						},
						AssumeRolePolicy: policy.IAMAssumeRolePolicy{
							Version: policy.DocumentVersion,
							Statement: []policy.IAMAssumeRoleStatement{
								policy.IAMAssumeRoleStatement{
									Effect: "Allow",
									Action: "sts:AssumeRole",
									Resource: []interface{}{
										putRoleResponse.RoleArn,
									},
								},
							},
						},
						AllowedArns: putRoleRequest.AllowedArns,
						WriteAccess: false,
						Owners:      owners,
					}
					So(putRoleResponse, ShouldResemble, expectedResponse)

				})
				Convey("with a valid role that doesn't have any roleOwners should work", func() {

					// Create a role
					tempRoleName := uuid.NewV4().String()
					tempRole := &policy.CreateRoleRequest{
						AllowedArns: []string{jchenArn},
						Name:        tempRoleName,
						SecretKeys: []string{
							testSecret1,
							testWildcardSecret,
							testWildcardSecret2,
						},
						WriteAccess: true,
						Owners:      []string{teamNavi},
					}
					bs, encodeErr := json.Marshal(tempRole)
					So(encodeErr, ShouldBeNil)
					resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusCreated)

					// Delete roleOwner
					pg := &policy.IAMPolicyGenerator{
						AuxPolicyArn:                 "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-aux",
						RWAuxPolicyArn:               "arn:aws:iam::734326455073:policy/sandstorm/testing/aux_policy/sandstorm-agent-testing-rw-aux",
						DynamoDBSecretsTableArn:      "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing",
						DynamoDBSecretsAuditTableArn: "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_audit",
						DynamoDBNamespaceTableArn:    "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing_namespaces",
						RoleOwnerTableName:           "sandstorm-testing_role_owners",
					}
					pg.SetPathPrefixEnvironment("testing")
					awsConfig := &aws.Config{Region: aws.String("us-west-2")}
					sess := session.New(awsConfig)
					stsclient := sts.New(sess)
					arp := &stscreds.AssumeRoleProvider{
						Duration:     900 * time.Second,
						ExpiryWindow: 10 * time.Second,
						RoleARN:      "arn:aws:iam::734326455073:role/sandstorm-apiserver-testing",
						Client:       stsclient,
					}
					credentials := credentials.NewCredentials(arp)
					awsConfig.WithCredentials(credentials)

					sess = session.New(awsConfig)
					pg.IAM = iam.New(sess)
					pg.DynamoDB = dynamodb.New(sess)

					err = pg.DeleteRoleOwners(tempRoleName, nil)
					So(err, ShouldBeNil)

					// try to edit as non-admin without owners
					tempRolePut := &policy.PutRoleRequest{
						SecretKeys:  []string{testSecret2},
						AllowedArns: []string{bachArn},
						WriteAccess: false,
					}

					bs, encodeErr = json.Marshal(tempRolePut)
					So(encodeErr, ShouldBeNil)
					resp, respErr = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRole.Name), "POST", bs, true)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					// cleanup
					resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRole.Name), "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
				})
				Convey("POST /roles/{existing-role} adding a secret that the user isnt allowed access should fail", func() {
					putRoleRequest := &policy.PutRoleRequest{
						SecretKeys:  []string{disallowedTestSecret},
						AllowedArns: []string{bachArn},
						WriteAccess: false,
						Owners:      []string{teamNavi},
					}
					bs, encodeErr := json.Marshal(putRoleRequest)
					So(encodeErr, ShouldBeNil)

					assembledURL := strings.Join([]string{baseURL, fmt.Sprintf("/roles/%s", role.Name)}, "")
					req, err := http.NewRequest("POST", assembledURL, bytes.NewBuffer(bs))
					So(err, ShouldBeNil)

					req.Header.Set("Origin", "http://localhost")
					req.Header.Set("Content-Type", contentType)
					// Setting access token for user that doesn't have access to syseng
					token := &oauth2.Token{AccessToken: LimitedAccessToken}
					token.SetAuthHeader(req)

					client := &http.Client{}
					failedResp, failerr := client.Do(req)
					So(failerr, ShouldBeNil)

					So(failedResp.StatusCode, ShouldEqual, http.StatusForbidden)

				})
				Convey("removing a secret that the user isnt allowed access should fail", func() {

					// Create role
					newRoleID := fmt.Sprintf("UnauthRemoveSecretTest-%s", uuid.NewV4().String())
					newRole := &policy.CreateRoleRequest{
						AllowedArns: []string{jchenArn},
						Name:        newRoleID,
						SecretKeys: []string{
							testSecret1,
							testSecret2,
						},
						WriteAccess: true,
						Owners:      []string{teamNavi},
					}
					newRoleMarshall, encodeErr := json.Marshal(newRole)
					So(encodeErr, ShouldBeNil)

					resp, respErr := testJSONRequest(baseURL, "/roles", "POST", newRoleMarshall, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusCreated)

					// Try to post without testSecret1 with user2 that doesn't have access
					putRoleRequest := &policy.PutRoleRequest{
						SecretKeys: []string{
							testSecret1,
						},
						AllowedArns: []string{jchenArn},
						WriteAccess: true,
					}
					failingPutRequest, encodeErr := json.Marshal(putRoleRequest)
					So(encodeErr, ShouldBeNil)

					assembledURL := strings.Join([]string{baseURL, fmt.Sprintf("/roles/%s", newRole.Name)}, "")
					req, err := http.NewRequest("POST", assembledURL, bytes.NewBuffer(failingPutRequest))
					So(err, ShouldBeNil)

					req.Header.Set("Origin", "http://localhost")
					req.Header.Set("Content-Type", contentType)
					// Setting access token for user that doesn't have access to syseng
					token := &oauth2.Token{AccessToken: LimitedAccessToken}
					token.SetAuthHeader(req)

					client := &http.Client{}
					failedResp, failerr := client.Do(req)
					So(failerr, ShouldBeNil)

					So(failedResp.StatusCode, ShouldEqual, http.StatusForbidden)

					// Delete role
					resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", newRole.Name), "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
				})
				Convey("Adding secrets when not a member but an admin should succeed", func() {
					// Create a role
					tempRoleName := uuid.NewV4().String()
					tempRole := &policy.CreateRoleRequest{
						AllowedArns: []string{jchenArn},
						Name:        tempRoleName,
						SecretKeys: []string{
							testSecret1,
							testWildcardSecret,
							testWildcardSecret2,
						},
						WriteAccess: true,
						Owners:      []string{teamNavi},
					}
					bs, encodeErr := json.Marshal(tempRole)
					So(encodeErr, ShouldBeNil)
					resp, respErr := testJSONRequest(baseURL, "/roles", "POST", bs, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusCreated)

					putRoleRequest := &policy.PutRoleRequest{
						SecretKeys: []string{
							testSecret1,
							testWildcardSecret,
							testWildcardSecret2,
							testSecret2,
						},
						AllowedArns: []string{jchenArn},
						Owners:      []string{teamNavi},
						WriteAccess: true,
					}
					bs, encodeErr = json.Marshal(putRoleRequest)
					So(encodeErr, ShouldBeNil)

					resp, respErr = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRoleName), "POST", bs, true)
					So(respErr, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					// Delete role
					resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", tempRoleName), "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
				})
			})

			Convey("DELETE /roles/{role}", func() {
				Convey("shouldn't delete the role", func() {
					Convey("when user isn't a roleOwner or Admin", func() {
						// Create role
						newRoleID := fmt.Sprintf("UnauthDeleteRoleTest-%s", uuid.NewV4().String())
						newRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Name:        newRoleID,
							SecretKeys: []string{
								testSecret1,
								testSecret2,
							},
							WriteAccess: true,
							Owners:      []string{"team-syseng"},
						}
						newRoleMarshall, encodeErr := json.Marshal(newRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", newRoleMarshall, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusCreated)

						// Attempt to delete role with non-member
						assembledURL := strings.Join([]string{baseURL, fmt.Sprintf("/roles/%s", newRole.Name)}, "")
						req, err := http.NewRequest("DELETE", assembledURL, nil)
						So(err, ShouldBeNil)

						req.Header.Set("Origin", "http://localhost")
						req.Header.Set("Content-Type", contentType)
						// Setting access token for user that doesn't have access to syseng
						token := &oauth2.Token{AccessToken: LimitedAccessToken}
						token.SetAuthHeader(req)

						client := &http.Client{}
						failedResp, failerr := client.Do(req)
						So(failerr, ShouldBeNil)
						So(failedResp.StatusCode, ShouldEqual, http.StatusForbidden)

						// Delete role
						resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", newRole.Name), "DELETE", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusNoContent)

					})
				})
				Convey("should delete the role", func() {
					Convey("If user is an admin but not an owner", func() {
						newRoleID := fmt.Sprintf("AdminDeleteRoleTest-%s", uuid.NewV4().String())
						newRole := &policy.CreateRoleRequest{
							AllowedArns: []string{jchenArn},
							Name:        newRoleID,
							SecretKeys: []string{
								testSecret1,
								testSecret2,
							},
							WriteAccess: true,
							Owners:      []string{"team-somebs"},
						}
						newRoleMarshall, encodeErr := json.Marshal(newRole)
						So(encodeErr, ShouldBeNil)

						resp, respErr := testJSONRequest(baseURL, "/roles", "POST", newRoleMarshall, true)
						So(respErr, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusCreated)

						// Delete role
						resp, err = testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", newRole.Name), "DELETE", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusNoContent)

					})
					Convey("given valid input", func() {
						resp, err := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", role.Name), "DELETE", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
					})
				})

				Convey("DELETE /roles/{nonexistent-role} should return 404", func() {
					resp, err := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", uuid.NewV4().String()), "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
				})
			})
		})
	})
}

func TestGroupAPI(t *testing.T) {
	cfg, err := LoadConfig("../test.hcl")
	if err != nil {
		t.Fatal(err)
	}

	tmpdir, err := ioutil.TempDir("", "changelogClient")
	if err != nil {
		t.Fatal(err)
	}

	defer func() {
		err := os.RemoveAll(tmpdir)
		if err != nil {
			t.Error(err)
		}
	}()

	logDir, err := ioutil.TempDir("", "logDir")
	if err != nil {
		t.Fatal(err)
	}

	defer func() {
		err := os.RemoveAll(logDir)
		if err != nil {
			t.Error(err)
		}
	}()

	cfg.Changelog.BoltDBFilePath = filepath.Join(tmpdir, "test.boltdb")
	cfg.Changelog.LogDir = logDir

	svc, err := New(cfg)
	if err != nil {
		t.Fatal(err)
	}
	server := httptest.NewServer(svc)
	defer server.Close()

	baseURL := server.URL

	assert := assert.New(t)

	type CreateGroup struct {
		RoleName string
	}

	var testRoleName = testutil.GetRandomGroupName()

	owners := []string{teamNavi}
	role := getCreateRoleRequest(jchenArn, testRoleName, true, owners)

	// patchGroupRequest holds groups name and write_access flag to update the groups.
	type patchGroupRequest struct {
		Name        string `json:"name,omitempty"`
		WriteAccess bool   `json:"write_access"`
	}

	t.Run("/groups api tests", func(t *testing.T) {

		createRoleJSON, encodeErr := json.Marshal(role)
		assert.Nil(encodeErr)

		// cleanup
		defer func() {
			_, err := testJSONRequest(baseURL, fmt.Sprintf("/roles/%s", testRoleName), "DELETE", nil, true)
			assert.Nil(err)
		}()

		resp, respErr := testJSONRequest(baseURL, "/roles", "POST", createRoleJSON, true)
		if assert.NoError(respErr) {
			assert.Equal(http.StatusCreated, resp.StatusCode)
		}

		createRoleResponse := new(policy.CreateRoleResponse)
		decodeErr := json.NewDecoder(resp.Body).Decode(createRoleResponse)
		assert.Nil(decodeErr)
		assert.Nil(resp.Body.Close())
		assert.NotEmpty(createRoleResponse.RoleArn)

		t.Run("group created along with role as the same name", func(t *testing.T) {
			createGroup := CreateGroup{RoleName: testRoleName}
			createJSON, err := json.Marshal(createGroup)
			assert.Nil(err)

			resp, err = testJSONRequest(baseURL, "/groups", "POST", createJSON, true)
			if assert.NoError(err) {
				assert.Equal(http.StatusOK, resp.StatusCode)
			}

			groupURI := fmt.Sprintf("/groups/%s", testRoleName)
			resp, err = testJSONRequest(baseURL, groupURI, "GET", nil, true)
			if assert.NoError(err) {
				assert.Equal(http.StatusOK, resp.StatusCode)
				assert.NotNil(resp.Body)
			}

			t.Run("Updating write access on group.", func(t *testing.T) {

				groupName := testRoleName
				groupNameURL := fmt.Sprintf("/groups/%s", groupName)

				resp, err = testJSONRequest(baseURL, groupNameURL, "GET", nil, true)
				_, err = ioutil.ReadAll(resp.Body)
				if assert.NoError(err) {
					assert.Equal(http.StatusOK, resp.StatusCode)
				}

				t.Run("Changing write access flag", func(t *testing.T) {
					testMap := make(map[string]bool)
					testMap[testRWAuxPolicyArn] = true
					testMap[testAuxPolicyArn] = false

					for policyArn, writeAccess := range testMap {

						patchGroup := patchGroupRequest{WriteAccess: writeAccess, Name: groupName}
						patchGroupJSON, err := json.Marshal(patchGroup)
						assert.Nil(err, "Json Marshal failed to marshal patchGroupRequest")

						resp, err = testJSONRequest(baseURL, groupNameURL, "PATCH", patchGroupJSON, true)
						if assert.NoError(err) {
							assert.Equal(http.StatusOK, resp.StatusCode, "Incorrect response code returned by 'PATCH /GROUPS/{groupName}'")
							htmlData, err := ioutil.ReadAll(resp.Body)
							assert.Nil(err)
							groupResponse := string(htmlData)
							assert.Contains(groupResponse, policyArn, "Group should be attached to read only aux policy")
						}

						resp, err = testJSONRequest(baseURL, groupNameURL, "GET", nil, true)
						assert.Nil(err, "GET /GROUPS/{groupName} failed")

						if assert.NoError(err) {
							assert.Equal(http.StatusOK, resp.StatusCode)
							htmlData, err := ioutil.ReadAll(resp.Body)
							assert.Nil(err)
							groupResponse := string(htmlData)
							assert.Contains(groupResponse, policyArn, "Group should be attached to read only aux policy")
						}
					}
				})
			})

			// cleanup
			resp, err := testJSONRequest(baseURL, fmt.Sprintf("/groups/%s", testRoleName), "DELETE", nil, true)
			if assert.NoError(err) {
				assert.Equal(http.StatusNoContent, resp.StatusCode)
			}
		})

		t.Run("deleting a group does not delete policy attached to role", func(t *testing.T) {
			createGroup := CreateGroup{RoleName: testRoleName}

			createJSON, err := json.Marshal(createGroup)
			assert.NoError(err)

			_, err = testJSONRequest(baseURL, "/groups", "POST", createJSON, true)
			assert.NoError(err)

			//clean up
			resp, err = testJSONRequest(baseURL, fmt.Sprintf("/groups/%s", testRoleName), "DELETE", nil, true)
			if assert.NoError(err) {
				assert.Equal(http.StatusNoContent, resp.StatusCode)
			}

			// Ensure that deleting group does not delete the policy
			attachedPolicies, err := getPolicyAttachedToRole(testRoleName)
			if assert.NoError(err) {
				assert.Len(attachedPolicies.AttachedPolicies, 2) // 1 for aux polixy and 1 for templated policy
			}
		})

		t.Run("Update aux policy on non existing group does not return error", func(t *testing.T) {

			groupName := "someRandomNonExistingGroup"
			patchGroup := patchGroupRequest{WriteAccess: true, Name: groupName}
			patchGroupJSON, err := json.Marshal(patchGroup)
			assert.Nil(err, "Json Marshal failed to marshal patchGroupRequest")
			groupNameURL := fmt.Sprintf("/groups/%s", groupName)

			resp, err = testJSONRequest(baseURL, groupNameURL, "PATCH", patchGroupJSON, true)
			if assert.NoError(err) {
				assert.Equal(http.StatusNoContent, resp.StatusCode, "Incorrect response code returned by 'PATCH /GROUPS/{groupName}'")
			}
		})
	})
}
