// +build integration

package apiserver

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"net/url"
	"os"
	"path/filepath"
	"strings"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"code.justin.tv/systems/sandstorm/testutil"

	. "github.com/smartystreets/goconvey/convey"
)

func TestSecretAPI(t *testing.T) {
	// Build basic url params
	testSecret := createTestSecret()

	Convey("API Tests for /secrets", t, func() {

		m := createTestManager()
		tmpdir, err := ioutil.TempDir("", "changelogClient")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(tmpdir)
			if err != nil {
				t.Error(err)
			}
		}()

		logDir, err := ioutil.TempDir("", "logDir")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(logDir)
			if err != nil {
				t.Error(err)
			}
		}()

		cfg, err := LoadConfig("../test.hcl")
		if err != nil {
			t.Fatal(err)
		}

		cfg.Changelog.BoltDBFilePath = filepath.Join(tmpdir, "test.boltdb")
		cfg.Changelog.LogDir = logDir

		svc, err := New(cfg)
		if err != nil {
			t.Fatal(err)
		}
		server := httptest.NewServer(svc)
		defer server.Close()

		copyName := copySecretName()

		baseURL := server.URL
		mgr := svc.clients.mgr.(*manager.Manager)

		// prepare URL for <secret> param
		secretURI := url.QueryEscape(testSecret.Name)

		Convey("Secret Handling Tests", func() {

			Convey("OAUTH", func() {

				Convey("should 401 if no OAuth Creds Provided", func() {
					resp, err := testJSONRequest(baseURL, "/secrets", "GET", nil, false)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusUnauthorized)
					So(resp.Body, ShouldNotBeNil)
				})
			})

			Convey("POST /secrets", func() {

				secretJSON, err := testSecretToJSON(testSecret)
				So(err, ShouldBeNil)

				Convey("should successfully create a new secret", func() {
					resp, err := testJSONRequest(baseURL, "/secrets", "POST", secretJSON, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					jsonSecret, err := secretFromBody(resp.Body)
					So(jsonSecret.Plaintext, ShouldBeNil)

					// TODO: scaffold things properly rather than this "hack"
					jsonSecret.Plaintext = testSecret.Plaintext
					testSecret = jsonSecret
				})

				Convey("should not override an existing secret", func() {
					resp, err := testJSONRequest(baseURL, "/secrets", "POST", secretJSON, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusConflict)
				})
			})

			Convey("PUT /secrets/<secret>", func() {
				Convey("updating cross_env should not be allowed for non sec users", func() {
					crossEnv := true

					body, err := json.Marshal(&manager.PatchInput{
						CrossEnv: &crossEnv,
					})
					So(err, ShouldBeNil)

					resp, err := testJSONRequest(baseURL, "/secrets/"+secretURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusUnauthorized)
				})
			})

			Convey("GET /secrets/<secret>", func() {

				Convey("should retrieve an existing secret", func() {
					getURI := strings.Join([]string{"/secrets", secretURI}, "/")
					resp, err := testJSONRequest(baseURL, getURI, "GET", nil, true)

					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)
					So(resp.Body, ShouldNotBeNil)

					secret, err := secretFromBody(resp.Body)
					So(err, ShouldBeNil)
					So(secret, ShouldNotBeNil)
				})

				Convey("should 404 non-url-encoded secret", func() {
					getURI := fmt.Sprintf("/secrets/%s", testSecret.Name)
					resp, err := testJSONRequest(baseURL, getURI, "GET", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
					So(resp.Body, ShouldNotBeNil)
				})

				Convey("should 404 on missing secret", func() {
					getURI := "/secrets/missing"
					resp, err := testJSONRequest(baseURL, getURI, "GET", nil, true)

					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
					So(resp.Body, ShouldNotBeNil)
				})
			})

			Convey("PATCH /secrets/<secret>", func() {

				Convey("should update an existing secret", func() {
					const newPlaintext = "PATCHTESTwootootoot123"

					testSecret.Name = copySecretName()
					secretURI = url.QueryEscape(testSecret.Name)
					secretJSON, err := testSecretToJSON(testSecret)
					resp, err := testJSONRequest(baseURL, "/secrets", "POST", secretJSON, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					dbSecret, err := mgr.Get(testSecret.Name)
					So(err, ShouldBeNil)
					So(dbSecret, ShouldNotBeNil)
					So(int8(dbSecret.Class), ShouldEqual, manager.ClassDefault)

					body, err := json.Marshal(&manager.PatchInput{
						Plaintext: []byte(newPlaintext),
					})
					So(err, ShouldBeNil)

					patchURI := strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err = testJSONRequest(baseURL, patchURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					dbSecret, err = mgr.Get(testSecret.Name)
					So(err, ShouldBeNil)
					So(string(dbSecret.Plaintext), ShouldEqual, newPlaintext)
					So(int8(dbSecret.Class), ShouldEqual, manager.ClassDefault)

					jsonSecret, err := secretFromBody(resp.Body)
					So(err, ShouldBeNil)
					So(jsonSecret.UpdatedAt, ShouldBeGreaterThanOrEqualTo, testSecret.UpdatedAt)
				})

				Convey("should update class on existing secret", func() {
					cls := manager.ClassCustomer
					body, err := json.Marshal(&manager.PatchInput{
						Class: &cls,
					})
					So(err, ShouldBeNil)

					patchURI := strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err := testJSONRequest(baseURL, patchURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					dbSecret, err := mgr.Get(testSecret.Name)
					So(err, ShouldBeNil)

					So(int8(dbSecret.Class), ShouldEqual, manager.ClassCustomer)

					jsonSecret, err := secretFromBody(resp.Body)
					So(err, ShouldBeNil)
					So(dbSecret.UpdatedAt, ShouldBeLessThanOrEqualTo, jsonSecret.UpdatedAt)
				})

				Convey("should not update class and do_not_broadcast flag if only secret is being changed", func() {
					cls := manager.ClassInternal
					dnb := true
					body, err := json.Marshal(&manager.PatchInput{
						Class:          &cls,
						DoNotBroadcast: &dnb,
					})
					So(err, ShouldBeNil)

					patchURI := strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err := testJSONRequest(baseURL, patchURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					dbSecret, err := mgr.Get(testSecret.Name)
					So(err, ShouldBeNil)
					So(dbSecret.Class, ShouldEqual, manager.ClassInternal)
					So(dbSecret.DoNotBroadcast, ShouldBeTrue)

					body, err = json.Marshal(&manager.PatchInput{
						Plaintext: []byte("newPlainText"),
					})

					patchURI = strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err = testJSONRequest(baseURL, patchURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					Convey("do_not_broadcast and class field was set correctly", func() {
						newDbSecret, err := mgr.Get(testSecret.Name)
						So(err, ShouldBeNil)

						So(newDbSecret.Class, ShouldEqual, manager.ClassInternal)
						So(newDbSecret.DoNotBroadcast, ShouldBeTrue)
					})

				})

				Convey("should gracefully handle attempting to patch a non-entity", func() {
					newSecret := createTestSecret()

					body, err := json.Marshal(&manager.PatchInput{
						Name: newSecret.Name,
					})
					So(err, ShouldBeNil)

					newSecretURI := url.QueryEscape(newSecret.Name)
					newPatchURI := strings.Join([]string{"/secrets", newSecretURI}, "/")

					resp, err := testJSONRequest(baseURL, newPatchURI, "PATCH", body, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
				})
			})

			Convey("PUT /secrets/<secret>/copy?destination=<dest>", func() {
				putURI := fmt.Sprintf("/secrets/%s/copy?destination=%s",
					url.QueryEscape(testSecret.Name),
					url.QueryEscape(copyName))

				Convey("should copy a source secret to a destination", func() {
					resp, err := testJSONRequest(baseURL, putURI, "PUT", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)
					err = mgr.Delete(copyName)
					So(err, ShouldBeNil)
				})

				Convey("New secret plaintext should equal old plaintext", func() {
					resp, err := testJSONRequest(baseURL, putURI, "PUT", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusOK)

					oldSecret, err := mgr.Get(testSecret.Name)
					So(err, ShouldBeNil)
					So(oldSecret, ShouldNotBeNil)

					newSecret, err := mgr.Get(copyName)
					So(err, ShouldBeNil)
					So(newSecret, ShouldNotBeNil)
					So(oldSecret.Plaintext, ShouldResemble, newSecret.Plaintext)

					err = mgr.Delete(newSecret.Name)
					So(err, ShouldBeNil)
				})

				Convey("should not panic if source does not exist.", func() {
					putURI := fmt.Sprintf("/secrets/%s/copy?destination=%s",
						url.QueryEscape("SecretDoesNotExist"),
						url.QueryEscape(copyName))

					resp, err := testJSONRequest(baseURL, putURI, "PUT", nil, true)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
				})
			})

			Convey("DELETE /secrets/<secret>", func() {

				Convey("should delete a secret", func() {
					deleteURI := strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err := testJSONRequest(baseURL, deleteURI, "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNoContent)

					deletedSecret, err := mgr.Get(testSecret.Name)
					So(deletedSecret, ShouldBeNil)
					So(err, ShouldBeNil)
				})

				Convey("should gracefully handle trying to delete a non-existent secret", func() {
					deleteURI := strings.Join([]string{"/secrets", secretURI}, "/")

					resp, err := testJSONRequest(baseURL, deleteURI, "DELETE", nil, true)
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusNoContent)
				})
			})

			Convey("GET /secrets/<secret>/versions", func() {
				versions := SecretVersionsPage{}

				Convey("with several versions already created", func() {
					testSecret := manager.Secret{
						Name:      testutil.GetRandomSecretName(),
						Plaintext: []byte("secretValue"),
					}
					So(m.Post(&testSecret), ShouldBeNil)
					// Update several times
					for n := 0; n < 3; n++ {
						// timestamp are accurate to the second
						time.Sleep(time.Duration(1 * time.Second))
						So(m.Patch(&manager.PatchInput{
							Name:      testSecret.Name,
							Plaintext: []byte{0x20 + byte(n)},
						}), ShouldBeNil)
					}

					Convey("should return all versions up to a limit", func() {
						path := fmt.Sprintf("/secrets/%s/versions", url.QueryEscape(testSecret.Name))
						resp, err := testJSONRequest(baseURL, path, "GET", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusOK)
						So(json.NewDecoder(resp.Body).Decode(&versions), ShouldBeNil)
						So(len(versions.Data), ShouldEqual, 4)
					})

					Convey("should paginate correctly", func() {
						path := fmt.Sprintf("/secrets/%s/versions?limit=1", url.QueryEscape(testSecret.Name))
						resp, err := testJSONRequest(baseURL, path, "GET", nil, true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusOK)
						So(json.NewDecoder(resp.Body).Decode(&versions), ShouldBeNil)
						So(len(versions.Data), ShouldEqual, 1)
						lastUpdatedAt := versions.Data[0].Attributes.UpdatedAt

						for n := 0; n < 3; n++ {
							path = fmt.Sprintf(
								"/secrets/%s/versions?limit=1&offset_key=%d",
								url.QueryEscape(testSecret.Name),
								versions.NextKey,
							)
							resp, err = testJSONRequest(baseURL, path, "GET", nil, true)
							So(err, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusOK)
							So(json.NewDecoder(resp.Body).Decode(&versions), ShouldBeNil)
							So(len(versions.Data), ShouldEqual, 1)
							// should be in descending order
							So(lastUpdatedAt, ShouldBeGreaterThan, versions.Data[0].Attributes.UpdatedAt)
							lastUpdatedAt = versions.Data[0].Attributes.UpdatedAt
						}
						So(versions.NextKey, ShouldEqual, 0)
					})

					Reset(func() {
						So(m.Delete(testSecret.Name), ShouldBeNil)
					})
				})
			})

			Convey("PUT /secrets/{secret}/revert", func() {
				Convey("with valid secret", func() {
					secretName := testutil.GetRandomSecretName()
					secretValues := []string{"secretValue-0", "secretValue-1"}
					revertPath := fmt.Sprintf("/secrets/%s/revert", url.QueryEscape(secretName))

					So(m.Post(&manager.Secret{
						Name:      secretName,
						Plaintext: []byte(secretValues[0]),
					}), ShouldBeNil)
					// Versions have second accuracy
					time.Sleep(time.Duration(1 * time.Second))
					So(m.Put(&manager.Secret{
						Name:      secretName,
						Plaintext: []byte(secretValues[1]),
					}), ShouldBeNil)

					versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
					So(err, ShouldBeNil)
					So(len(versions.Secrets), ShouldEqual, 2)

					Convey("success case", func() {
						firstVersion := versions.Secrets[1].UpdatedAt

						resp, err := testJSONRequest(
							baseURL,
							revertPath,
							"PUT",
							[]byte(fmt.Sprintf("{\"version\": %d}", firstVersion)),
							true)
						So(err, ShouldBeNil)
						So(resp.StatusCode, ShouldEqual, http.StatusOK)

						currentVersion, err := m.Get(secretName)
						So(err, ShouldBeNil)
						So(string(currentVersion.Plaintext), ShouldEqual, secretValues[0])
					})

					Convey("invalid body should return 400", func() {
						Convey("no body", func() {
							resp, err := testJSONRequest(baseURL, revertPath, "PUT", nil, true)
							So(err, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
						})

						Convey("missing version", func() {
							resp, err := testJSONRequest(baseURL, revertPath, "PUT", []byte("{}"), true)
							So(err, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
						})

						Convey("invalid type", func() {
							resp, err := testJSONRequest(baseURL, revertPath, "PUT", []byte("{\"version\": \"derp\"}"), true)
							So(err, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
						})

						Convey("invalid version", func() {
							resp, err := testJSONRequest(baseURL, revertPath, "PUT", []byte("{\"version\": 0}"), true)
							So(err, ShouldBeNil)
							So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
						})
					})

					Reset(func() {
						So(m.Delete(secretName), ShouldBeNil)
					})
				})

				Convey("Invalid secret should return 400", func() {
					resp, err := testJSONRequest(baseURL, "/secrets/missing/revert", "PUT", nil, true)
					So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}
