// +build integration

package apiserver

import (
	"encoding/json"
	"net/http"
	"testing"

	"code.justin.tv/systems/sandstorm/cryptorand"
	"code.justin.tv/systems/sandstorm/manager"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestGetSecretVersions(t *testing.T) {
	t.Run("GET /secrets/<secret>/versions", func(t *testing.T) {
		t.Parallel()
		assert := assert.New(t)
		server, mocked := setupMockAPI(t)
		defer server.Close()
		baseURL := server.URL

		t.Run("should return no versions for a missing secret", func(t *testing.T) {
			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.mgr.On("GetVersionsEncrypted", "missing", int64(10), int64(0)).Return(&manager.VersionsPage{
				Secrets: []*manager.Secret{},
				Limit:   10,
			}, nil)
			resp, err := testJSONRequest(baseURL, "/secrets/missing/versions", "GET", nil, true)
			if !assert.NoError(err) {
				t.FailNow()
			}

			assert.True(mocked.mgr.AssertExpectations(t))
			assert.True(mocked.authorizer.AssertExpectations(t))

			if assert.NotNil(resp) {
				assert.Equal(http.StatusOK, resp.StatusCode)

				var versions SecretVersionsPage
				assert.NoError(json.NewDecoder(resp.Body).Decode(&versions))
				expected := SecretVersionsPage{
					Data:      []SecretElement{},
					Limit:     10,
					NextKey:   0,
					OffsetKey: 0,
				}

				assert.EqualValues(expected, versions)
			}
		})
	})
}

func TestPostSecrets(t *testing.T) {
	assert := assert.New(t)

	t.Run("POST /secrets", func(t *testing.T) {
		secretName := "mySecretName"
		secretVersion := int64(48)

		t.Run("should autogenerate plaintext if given autogenerate_plaintext", func(t *testing.T) {
			t.Parallel()
			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			var postedSecret *manager.Secret

			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.mgr.On("Post", mock.Anything).Run(
				func(args mock.Arguments) {
					// Store the posted secret so we can check what was sent
					postedSecret = args.Get(0).(*manager.Secret)
				},
			).Return(nil).Once()
			mocked.mgr.On("Exist", secretName).Return(false, nil).Once()
			mocked.mgr.On("GetEncrypted", secretName).Return(&manager.Secret{
				Name:      secretName,
				UpdatedAt: secretVersion,
			}, nil).Once()

			body, err := json.Marshal(&putSecretRequest{
				Data: SecretElement{
					Attributes: &manager.Secret{Name: secretName},
				},
				Autogenerate: manager.FillPlaintextRequest{Length: 28},
			})
			assert.Nil(err)

			resp, err := testJSONRequest(baseURL, "/secrets", "POST", body, true)
			assert.Nil(err)
			assert.Equal(http.StatusOK, resp.StatusCode)

			response := SecretSingle{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))
			assert.Equal(SecretSingle{
				Data: SecretElement{
					ID:   secretName,
					Type: "Secret",
					Attributes: &manager.Secret{
						Name:      secretName,
						UpdatedAt: secretVersion,
					},
				},
			}, response)

			assert.True(mocked.authorizer.AssertExpectations(t))
			assert.True(mocked.mgr.AssertExpectations(t))

			assert.Len(postedSecret.Plaintext, 40, "Created secret should have 40 characters after base64 encode.")
		})

		t.Run("should refuse cross_env as true", func(t *testing.T) {
			t.Parallel()
			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			mocked.authorizer.On("IsCrossEnvAdmin", mock.Anything).Return(false)

			body, err := json.Marshal(&putSecretRequest{
				Data: SecretElement{
					Attributes: &manager.Secret{
						Name:      secretName,
						CrossEnv:  true,
						Plaintext: []byte("mySecret"),
					},
				},
			})
			assert.Nil(err)

			resp, err := testJSONRequest(baseURL, "/secrets", "POST", body, true)
			assert.NoError(err)
			assert.Equal(http.StatusUnauthorized, resp.StatusCode)

			assert.True(mocked.authorizer.AssertExpectations(t))
			assert.True(mocked.mgr.AssertExpectations(t))
		})

		t.Run("cross_env ok if in group", func(t *testing.T) {
			t.Parallel()
			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.authorizer.On("IsCrossEnvAdmin", mock.Anything).Return(true)
			mocked.mgr.On("Post", mock.Anything).Return(nil).Once()
			mocked.mgr.On("Exist", secretName).Return(false, nil).Once()
			mocked.mgr.On("GetEncrypted", secretName).Return(&manager.Secret{
				Name:      secretName,
				UpdatedAt: secretVersion,
			}, nil).Once()

			body, err := json.Marshal(&putSecretRequest{
				Data: SecretElement{
					Attributes: &manager.Secret{
						Name:      secretName,
						CrossEnv:  true,
						Plaintext: []byte("myPlaintext"),
					},
				},
			})
			assert.Nil(err)

			resp, err := testJSONRequest(baseURL, "/secrets", "POST", body, true)
			assert.Nil(err)
			assert.Equal(http.StatusOK, resp.StatusCode)

			response := SecretSingle{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))
			assert.Equal(SecretSingle{
				Data: SecretElement{
					ID:   secretName,
					Type: "Secret",
					Attributes: &manager.Secret{
						Name:      secretName,
						UpdatedAt: secretVersion,
					},
				},
			}, response)

			assert.True(mocked.authorizer.AssertExpectations(t))
			assert.True(mocked.mgr.AssertExpectations(t))
		})

		t.Run("should refuse request if both autogenerate and plaintext are sent", func(t *testing.T) {
			t.Parallel()
			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			body, err := json.Marshal(&putSecretRequest{
				Data: SecretElement{
					Attributes: &manager.Secret{
						Name:      secretName,
						Plaintext: []byte("myPlaintext"),
					},
				},
				Autogenerate: manager.FillPlaintextRequest{Length: 28},
			})
			assert.Nil(err)

			resp, err := testJSONRequest(baseURL, "/secrets", "POST", body, true)
			assert.Nil(err)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)

			assert.True(mocked.authorizer.AssertExpectations(t))
		})

		t.Run("should refuse autogenerate plaintext if autogenerate_plaintext_bytes is less than 2", func(t *testing.T) {
			t.Parallel()
			server, _ := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			body, err := json.Marshal(&putSecretRequest{
				Data: SecretElement{
					Attributes: &manager.Secret{Name: secretName},
				},
				Autogenerate: manager.FillPlaintextRequest{Length: 1},
			})
			assert.Nil(err)

			resp, err := testJSONRequest(baseURL, "/secrets", "POST", body, true)
			assert.Nil(err)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)
		})
	})
}

func TestPatchSecret(t *testing.T) {
	t.Run("PATCH /secrets/<secret>", func(t *testing.T) {
		assert := assert.New(t)
		secretName := "mySecretName"
		secretVersion := int64(48)

		t.Run("should update with autogenerated secret", func(t *testing.T) {
			t.Parallel()

			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			dnb := true
			cls := manager.ClassRestricted

			req := &manager.PatchInput{
				Name:           secretName,
				DoNotBroadcast: &dnb,
				Class:          &cls,
				Autogenerate: &manager.FillPlaintextRequest{
					Length: 28,
				},
			}
			// want to compare validated version
			assert.NoError(req.Validate())

			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.mgr.On("Patch", req).Return(nil).Once()
			mocked.mgr.On("GetEncrypted", secretName).Return(&manager.Secret{
				Name:      secretName,
				UpdatedAt: secretVersion,
			}, nil).Once()

			body, err := json.Marshal(req)
			if !assert.NoError(err) {
				t.Fatal(err)
			}

			resp, err := testJSONRequest(baseURL, "/secrets/"+secretName, "PATCH", body, true)
			if !assert.NoError(err) {
				t.Fatal(err)
			}
			assert.Equal(resp.StatusCode, http.StatusOK)

			response := SecretSingle{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))
			assert.Equal(SecretSingle{
				Data: SecretElement{
					ID:   secretName,
					Type: "Secret",
					Attributes: &manager.Secret{
						Name:      secretName,
						UpdatedAt: secretVersion,
					},
				},
			}, response)

			assert.True(mocked.mgr.AssertExpectations(t))
			assert.True(mocked.authorizer.AssertExpectations(t))
		})

		t.Run("patch cross_env to false is ok", func(t *testing.T) {
			t.Parallel()

			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			crossEnv := false

			req := &manager.PatchInput{
				Name:     secretName,
				CrossEnv: &crossEnv,
			}
			// want to compare validated version
			assert.NoError(req.Validate())

			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.mgr.On("Patch", req).Return(nil).Once()
			mocked.mgr.On("GetEncrypted", secretName).Return(&manager.Secret{
				Name:      secretName,
				UpdatedAt: secretVersion,
			}, nil).Once()

			body, err := json.Marshal(req)
			if !assert.NoError(err) {
				t.Fatal(err)
			}

			resp, err := testJSONRequest(baseURL, "/secrets/"+secretName, "PATCH", body, true)
			if !assert.NoError(err) {
				t.Fatal(err)
			}
			assert.Equal(http.StatusOK, resp.StatusCode)

			response := SecretSingle{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))
			assert.Equal(SecretSingle{
				Data: SecretElement{
					ID:   secretName,
					Type: "Secret",
					Attributes: &manager.Secret{
						Name:      secretName,
						UpdatedAt: secretVersion,
					},
				},
			}, response)

			assert.True(mocked.mgr.AssertExpectations(t))
			assert.True(mocked.authorizer.AssertExpectations(t))
		})

		t.Run("should return 401 if cross_env is set to true", func(t *testing.T) {
			t.Parallel()

			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			crossEnv := true

			req := &manager.PatchInput{
				Name:     secretName,
				CrossEnv: &crossEnv,
			}

			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.authorizer.On("IsCrossEnvAdmin", mock.Anything).Return(false)

			body, err := json.Marshal(req)
			if !assert.NoError(err) {
				t.Fatal(err)
			}

			resp, err := testJSONRequest(baseURL, "/secrets/"+secretName, "PATCH", body, true)
			if !assert.NoError(err) {
				t.Fatal(err)
			}
			assert.Equal(http.StatusUnauthorized, resp.StatusCode)

			assert.True(mocked.mgr.AssertExpectations(t))
			assert.True(mocked.authorizer.AssertExpectations(t))
		})

		t.Run("patch cross_env to true is ok if CrossEnv admin", func(t *testing.T) {
			t.Parallel()

			server, mocked := setupMockAPI(t)
			defer server.Close()
			baseURL := server.URL

			crossEnv := true

			req := &manager.PatchInput{
				Name:     secretName,
				CrossEnv: &crossEnv,
			}
			// want to compare validated version
			assert.NoError(req.Validate())

			mocked.authorizer.On("IsCrossEnvAdmin", mock.Anything).Return(true)
			mocked.authorizer.On("IsAllowed", mock.Anything, mock.Anything).Return(nil)
			mocked.mgr.On("Patch", req).Return(nil).Once()
			mocked.mgr.On("GetEncrypted", secretName).Return(&manager.Secret{
				Name:      secretName,
				UpdatedAt: secretVersion,
			}, nil).Once()

			body, err := json.Marshal(req)
			if !assert.NoError(err) {
				t.Fatal(err)
			}

			resp, err := testJSONRequest(baseURL, "/secrets/"+secretName, "PATCH", body, true)
			if !assert.NoError(err) {
				t.Fatal(err)
			}
			assert.Equal(http.StatusOK, resp.StatusCode)

			response := SecretSingle{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))
			assert.Equal(SecretSingle{
				Data: SecretElement{
					ID:   secretName,
					Type: "Secret",
					Attributes: &manager.Secret{
						Name:      secretName,
						UpdatedAt: secretVersion,
					},
				},
			}, response)

			assert.True(mocked.mgr.AssertExpectations(t))
			assert.True(mocked.authorizer.AssertExpectations(t))
		})
	})
}

func TestPostGeneratePlaintext(t *testing.T) {
	assert := assert.New(t)
	server, _ := setupMockAPI(t)
	defer server.Close()
	baseURL := server.URL

	t.Run("POST /generate-plaintext", func(t *testing.T) {
		t.Run("should successfully return plaintext", func(t *testing.T) {
			t.Parallel()

			generatePlaintextRequest := cryptorand.GeneratePlaintextRequest{
				IncludeAlphaLowercase: true,
				Length:                20,
			}

			body, err := json.Marshal(&generatePlaintextRequest)
			assert.Nil(err)

			resp, respErr := testJSONRequest(baseURL, "/generate-plaintext", "POST", body, true)
			assert.Nil(respErr)
			assert.NotNil(resp)
			assert.Equal(http.StatusOK, resp.StatusCode)

			response := postGeneratePlaintextResponse{}
			assert.Nil(json.NewDecoder(resp.Body).Decode(&response))

			assert.Len(response.Plaintext, 20)
			for _, b := range []byte(response.Plaintext) {
				assert.Contains([]byte("abcdefghijklmnopqrstuvwxyz"), b)
			}
		})

		t.Run("should refuse all includes false", func(t *testing.T) {
			t.Parallel()

			generatePlaintextRequest := cryptorand.GeneratePlaintextRequest{
				Length: 20,
			}

			body, err := json.Marshal(&generatePlaintextRequest)
			assert.Nil(err)

			resp, respErr := testJSONRequest(baseURL, "/generate-plaintext", "POST", body, true)
			assert.Nil(respErr)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)
		})

		t.Run("should refuse length less than 2", func(t *testing.T) {
			t.Parallel()

			generatePlaintextRequest := cryptorand.GeneratePlaintextRequest{
				IncludeAlphaLowercase: true,
				Length:                1,
			}

			body, err := json.Marshal(&generatePlaintextRequest)
			assert.Nil(err)

			resp, respErr := testJSONRequest(baseURL, "/generate-plaintext", "POST", body, true)
			assert.Nil(respErr)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)
		})

		t.Run("should refuse length 0", func(t *testing.T) {
			t.Parallel()

			generatePlaintextRequest := cryptorand.GeneratePlaintextRequest{
				IncludeAlphaLowercase: true,
			}

			body, err := json.Marshal(&generatePlaintextRequest)
			assert.Nil(err)

			resp, respErr := testJSONRequest(baseURL, "/generate-plaintext", "POST", body, true)
			assert.Nil(respErr)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)
		})

		t.Run("should refuse length greater than 4096", func(t *testing.T) {
			t.Parallel()

			generatePlaintextRequest := cryptorand.GeneratePlaintextRequest{
				IncludeAlphaLowercase: true,
				Length:                4097,
			}

			body, err := json.Marshal(&generatePlaintextRequest)
			assert.Nil(err)

			resp, respErr := testJSONRequest(baseURL, "/generate-plaintext", "POST", body, true)
			assert.Nil(respErr)
			assert.Equal(http.StatusBadRequest, resp.StatusCode)
		})
	})
}
